# -*- coding: utf-8 -*-
from __future__ import absolute_import

from abc import ABCMeta, abstractproperty
from typing import List  # noqa
from travel.avia.library.python.common.models.transport import TransportModel


class TransportModelDTO(object):
    __slots__ = (
        'title', 'title_en', 'code', 'code_en', 'is_cargo',
        'is_propeller_flight', 'plane_body_type', 'producer',
    )

    def __init__(self, title, title_en, code, code_en, is_cargo,
                 is_propeller_flight, plane_body_type, producer):
        self.title = title
        self.title_en = title_en
        self.code = code
        self.code_en = code_en
        self.is_cargo = is_cargo
        self.is_propeller_flight = is_propeller_flight
        self.plane_body_type = plane_body_type
        self.producer = producer


class TransportModelRepository(object):
    __metaclass__ = ABCMeta

    def __init__(self):
        self._type = abstractproperty()
        self._by_code_en = {}

    def pre_cache(self):
        # type: () -> None
        models = self._load_models()

        self._by_code_en = {
            model['code_en']: TransportModelDTO(
                model['title'],
                model['title_en'],
                model['code'],
                model['code_en'],
                model['is_cargo'],
                model['is_propeller_flight'],
                model['plane_body_type'],
                model['producer__title'],
            ) for model in models
        }

    def _load_models(self):
        # type: () -> List[dict]
        return TransportModel.objects.values(
            'title', 'title_en', 'code', 'code_en', 'is_cargo', 'is_propeller_flight',
            'plane_body_type', 'producer__title',
        ).filter(t_type__code=self._type)

    def get_by_code_en(self, code_en):
        # type: (str) -> TransportModelDTO
        return self._by_code_en.get(code_en)

    def get_all(self):
        # type: () -> List[TransportModelDTO]
        return self._by_code_en.values()


class PlaneTransportModelRepository(TransportModelRepository):
    def __init__(self):
        super(PlaneTransportModelRepository, self).__init__()
        self._type = 'plane'


plane_transport_model_repository = PlaneTransportModelRepository()
