package main

import (
	"context"
	"fmt"
	"net/http"
	"sync"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/avia/chatbot/internal/messages"
	"a.yandex-team.ru/travel/avia/chatbot/internal/settings"
	"a.yandex-team.ru/travel/avia/chatbot/internal/storage/history/flight"
	"a.yandex-team.ru/travel/avia/chatbot/public/botapi"
	"a.yandex-team.ru/travel/avia/chatbot/public/pricesub"
	priceSubImpl "a.yandex-team.ru/travel/avia/chatbot/public/pricesub/impl"
	searchResult "a.yandex-team.ru/travel/avia/chatbot/public/search_result"
	"a.yandex-team.ru/travel/avia/chatbot/public/sharedflights"
	result "a.yandex-team.ru/travel/avia/library/proto/search_result/v1"
	"a.yandex-team.ru/travel/library/go/configuration"
)

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	configLoader := configuration.NewDefaultConfitaLoader()

	config := settings.Config{}
	err := configLoader.Load(ctx, &config)

	if err != nil {
		fmt.Printf("Cannot load application config: %v", err)
		ctxCancel()
		return
	}

	logger, err := zap.NewDeployLogger(log.InfoLevel)
	if err != nil {
		fmt.Printf("Cannot load application logger: %v", err)
		return
	}

	commonChannel := make(chan *result.Result)
	searchResult.StartMultiReader(
		ctx, []string{
			"iva.logbroker.yandex.net",
			"sas.logbroker.yandex.net",
			"vla.logbroker.yandex.net",
		}, config.Logbroker.Topic,
		config.Logbroker.Consumer,
		ydb.NewAuthTokenCredentials(config.Logbroker.Token, ""),
		commonChannel,
		logger,
	)
	logger.Info("Creating botapi Client")
	bot, err := botapi.NewClient(
		config.Telegram.Token,
		botapi.Logger(logger.WithName("botapi")),
		botapi.Whitelist(
			append(
				settings.HardcodedWhitelist,
				config.Telegram.Whitelist...,
			),
		),
	)
	if err != nil {
		fmt.Printf("Cannot create telebot api client: %v", err)
	}
	repository := priceSubImpl.NewInMemoryRepository()
	subscribeHandler := pricesub.SubscribeHandler{
		SubscriberSaver:             repository,
		SubscriptionSaver:           repository,
		SubscriberSubscriptionSaver: repository,
	}

	wg := sync.WaitGroup{}
	wg.Add(1)
	go func() {
		defer wg.Done()
		telegramSender := &priceSubImpl.TelegramSender{
			Bot:    bot.Bot,
			Logger: logger.WithName("PriceSubscriptionTelegramSender"),
		}
		priceSubImpl.ChannelLoop(
			ctx,
			logger.WithName("priceSubscriptionLoop"),
			repository,
			repository,
			telegramSender,
			telegramSender,
			commonChannel,
		)
	}()

	handlers := []botapi.MessageHandler{
		&messages.StartHandler{},
		messages.NewFlightHandler(
			logger.WithName("flightHandler"),
			sharedflights.Client{
				Host: sharedflights.APIHost,
				HTTP: http.Client{Timeout: 10 * time.Second},
			},
			flight.NewFlightHistory(4),
		),
		messages.NewPriceSubscriptionHandler(
			logger.WithName("priceSubscription"),
			&subscribeHandler,
		),
	}

	if err != nil {
		logger.Errorf("Cannot create telegram bot: %v", err)
		return
	}
	for _, handler := range handlers {
		bot.AddHandler(handler)
	}
	wg.Add(1)
	go func() {
		defer wg.Done()
		bot.PollForever()
	}()
	wg.Wait()
}
