package messages

import (
	"fmt"
	"strings"
	"time"

	"gopkg.in/tucnak/telebot.v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/chatbot/public/botapi"
	"a.yandex-team.ru/travel/avia/chatbot/public/sharedflights"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
)

const MessageTimeout = 10 * time.Second

type FlightHistory interface {
	SaveFlight(FlightNumber string)
	GetHistory() []string
}

type FlightHandler struct {
	Log                 log.Logger
	SharedFlightsClient sharedflights.Client
	History             FlightHistory
}

func NewFlightHandler(Log log.Logger, SharedFlightsClient sharedflights.Client, History FlightHistory) *FlightHandler {
	return &FlightHandler{
		Log,
		SharedFlightsClient,
		History,
	}
}

func (f *FlightHandler) Handle(message telebot.Message) <-chan botapi.Response {
	ch := make(chan botapi.Response)
	// TODO: implement tokenization, because command can be anywhere in message
	message.Text = strings.TrimSpace(strings.TrimPrefix(message.Text, f.TelegramCommand()))
	if len(message.Text) == 0 {
		go func() {
			defer close(ch)

			history := f.History.GetHistory()
			if len(history) == 0 {
				_ = PutWithTimeout(ch, botapi.Response{
					Text: "Try: " + strings.Join([]string{f.TelegramCommand(), "SU 1404"}, " "),
				}, MessageTimeout)
				return
			}
			buttons := make([]botapi.ResponseButton, 0, len(history))
			for _, h := range history {
				buttons = append(buttons, botapi.ResponseButton{
					Text:    h,
					Command: strings.Join([]string{f.TelegramCommand(), h}, " "),
				})
			}
			_ = PutWithTimeout(ch, botapi.Response{
				Text:     "",
				Keyboard: buttons,
			}, MessageTimeout)
		}()
		return ch
	}
	got := strings.Split(message.Text, " ")
	var carrier, flightNumber, dateStr string
	switch len(got) {
	case 2:
		carrier, flightNumber = got[0], got[1]
	case 3:
		carrier, flightNumber, dateStr = got[0], got[1], got[2]
	default:
		go func() {
			defer close(ch)
			_ = PutWithTimeout(ch, botapi.Response{Text: "Некорректный формат. Попробуй `SU 1404` или `SU 1404 2022-10-02`"}, MessageTimeout)
		}()
		return ch
	}
	var dates []dtutil.StringDate

	if len(dateStr) == 0 {
		dates = []dtutil.StringDate{dtutil.FormatDateIso(time.Now()), dtutil.FormatDateIso(time.Now().AddDate(0, 0, 1))}
	} else {
		// TODO: invalid date format
		dates = []dtutil.StringDate{dtutil.StringDate(dateStr)}
	}
	go func() {
		defer close(ch)
		for _, d := range dates {
			msgToSend := f.getStatus(carrier, flightNumber, d)
			err := PutWithTimeout(ch, botapi.Response{Text: msgToSend}, MessageTimeout)
			if err != nil {
				return
			}
			f.Log.Info(msgToSend, log.Int64("for", message.Chat.ID))
		}
	}()
	f.History.SaveFlight(message.Text)
	return ch
}
func (f *FlightHandler) TelegramCommand() string {
	return "/f"
}
func (f *FlightHandler) ShouldHandle(m telebot.Message) bool {
	return strings.HasPrefix(m.Text, f.TelegramCommand()) || !strings.HasPrefix(m.Text, "/") && f.looksLikeFlightNumber(m.Text)
}

func (f *FlightHandler) looksLikeFlightNumber(text string) bool {
	runes := []rune(text)
	if !strings.Contains(text, " ") && (len(runes) < 4 || len(runes) > 7) {
		return false
	}
	if code, ok := f.SharedFlightsClient.FlightNumbersToCarriers([]string{text})[text]; ok && code != nil {
		return true
	}
	return false
}

func parseStatusDateTime(dt string) time.Time {
	if len(dt) == 0 {
		return time.Time{}
	}
	layout := "2006-01-02 15:04:05"
	t, _ := time.Parse(layout, dt)
	return t
}

func parseScheduleDateTime(strDate string, strTime string) time.Time {
	if len(strDate) == 0 || len(strTime) == 0 {
		return time.Time{}
	}
	datetime := strDate + "T" + strTime
	layout := "2006-01-02T15:04:05"
	t, _ := time.Parse(layout, datetime)
	return t
}

func (f *FlightHandler) humanReadableStatusString(status sharedflights.FlightStatusText) string {
	switch status {
	case sharedflights.FlightStatusArrived:
		return "прибыл"
	case sharedflights.FlightStatusCancelled:
		return "отменён"
	case sharedflights.FlightStatusDelayed:
		return "задерживается"
	case sharedflights.FlightStatusDeparted:
		return "вылетел"
	case sharedflights.FlightStatusEarly:
		return "раньше срока"
	case sharedflights.FlightStatusOnTime:
		return "по расписанию"
	case sharedflights.FlightStatusUnknown:
		return "статус неизвестен"
	default:
		return string(status)
	}
}

func (f *FlightHandler) getStatus(carrier string, number string, d dtutil.StringDate) string {
	resp := f.SharedFlightsClient.Flight(carrier, number, string(d))
	if len(resp.Number) == 0 {
		return fmt.Sprintf("%v %v нет рейса на дату %v", carrier, number, d)
	}

	flightStatus := f.humanReadableStatusString(resp.Status.Status)
	departureStatus := f.humanReadableStatusString(resp.Status.DepartureStatus)
	departureGate := resp.Status.DepartureGate
	checkinDesks := resp.Status.CheckInDesks
	departureTerminal := resp.Status.DepartureTerminal
	if len(departureTerminal) == 0 {
		departureTerminal = resp.DepartureTerminal
	}
	departureTime := parseStatusDateTime(resp.Status.DepartureDT)
	schDepartureTime := parseScheduleDateTime(resp.DepartureDay, resp.DepartureTime)

	if resp.Status.Status == sharedflights.FlightStatusArrived {
		return fmt.Sprintf("Рейс %v на %v %v", resp.Title, schDepartureTime.Format("02.01"), flightStatus)
	}

	hasStatus := len(resp.Status.Status) > 0 && resp.Status.Status != sharedflights.FlightStatusUnknown
	hasDepartureStatus := len(resp.Status.DepartureStatus) > 0 && resp.Status.DepartureStatus != sharedflights.FlightStatusUnknown
	hasGate := len(departureGate) > 0
	hasCheckin := len(checkinDesks) > 0
	hasTerminal := len(departureTerminal) > 0
	hasStatusTime := !departureTime.IsZero()
	hasScheduleTime := !schDepartureTime.IsZero()

	statusStr := flightStatus

	if hasDepartureStatus && resp.Status.DepartureStatus != resp.Status.Status {
		statusStr += fmt.Sprintf(" (%v)", departureStatus)
	}
	statusStr += "."

	if !hasStatus && !hasDepartureStatus {
		statusStr = "статус неизвестен."
	}

	terminalStr := fmt.Sprintf("Терминал %v.", departureTerminal)
	if !hasTerminal {
		terminalStr = ""
	}

	checkinStr := fmt.Sprintf("Стойки регистрации %v.", checkinDesks)
	if !hasCheckin {
		checkinStr = ""
	}

	gateStr := fmt.Sprintf("Выход на посадку %v.", departureGate)
	if !hasGate {
		gateStr = ""
	}

	statusTimeStrFormat := "ожидается в %v"
	if departureTime == schDepartureTime {
		statusTimeStrFormat = "ожидается по расписанию в %v"
	}
	statusTimeStr := fmt.Sprintf(statusTimeStrFormat, formatStatusTime(departureTime, schDepartureTime))
	if !hasStatusTime {
		statusTimeStr = ""
	}
	scheduleTimeStr := fmt.Sprintf("согласно расписанию в %v", schDepartureTime.Format("15:04"))
	if !hasScheduleTime || departureTime == schDepartureTime {
		scheduleTimeStr = ""
	}

	finalStatusString := strings.Join([]string{"Рейс", resp.Title, "на", schDepartureTime.Format("02.01"), statusStr, terminalStr, checkinStr, gateStr, "Вылет", statusTimeStr, scheduleTimeStr}, " ")
	for strings.Contains(finalStatusString, "  ") {
		finalStatusString = strings.Replace(
			finalStatusString,
			"  ",
			" ",
			-1,
		)
	}
	return finalStatusString
}

func formatStatusTime(statusTime time.Time, scheduleTime time.Time) string {
	if statusTime.Format("02.01") == scheduleTime.Format("02.01") {
		return statusTime.Format("15:04")
	}
	return statusTime.Format("02.01 15:04")
}
