package messages

import (
	"context"
	"fmt"
	"strings"
	"time"

	"gopkg.in/tucnak/telebot.v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/chatbot/public/botapi"
	"a.yandex-team.ru/travel/avia/chatbot/public/pricesub"
)

func NewPriceSubscriptionHandler(Log log.Logger, subscribeHandler *pricesub.SubscribeHandler) *PriceSubscriptionHandler {
	return &PriceSubscriptionHandler{
		logger:           Log,
		subscribeHandler: subscribeHandler,
	}
}

type PriceSubscriptionHandler struct {
	logger           log.Logger
	subscribeHandler *pricesub.SubscribeHandler
}

func subscribe(ctx context.Context, subscribeHandler *pricesub.SubscribeHandler, chatID int64, qKey string, filter pricesub.Filter) error {
	query, err := pricesub.QueryFromQkey(qKey)
	if err != nil {
		return err
	}
	subscribeHandler.Subscribe(
		ctx,
		pricesub.Subscriber{
			ChatID: chatID,
		},
		pricesub.Subscription{
			Query:  *query,
			Filter: filter,
		},
	)
	return nil
}

func (p PriceSubscriptionHandler) Handle(message telebot.Message) <-chan botapi.Response {
	parts := strings.Split(message.Text, " ")

	response := make(chan botapi.Response)
	if len(parts) >= 2 {
		go func(qkey string, options ...string) {
			defer close(response)
			ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
			defer cancel()
			filter := pricesub.Filter{}
			for _, option := range options {
				if option == "+багаж" || option == "+baggage" {
					filter.Baggage = ptr.Bool(true)
				} else if strings.HasPrefix(option, "-пересадк") ||
					option == "direct" {
					filter.MaxTransferCount = ptr.Int(0)
				}
			}
			if err := subscribe(ctx, p.subscribeHandler, message.Chat.ID, qkey, filter); err != nil {
				response <- botapi.Response{Text: fmt.Sprintf("Не могу подписать на поиск: %s", err)}
				return
			}
			response <- botapi.Response{Text: fmt.Sprintf("Успешно подписан на поиск %s", qkey)}
		}(parts[1], parts[2:]...)
	} else {
		go func() {
			defer close(response)
			response <- botapi.Response{Text: "Некорректный формат: используй `/s c213_c239_2021-10-01_None_1_0_0_economy_ru`"}
		}()
	}
	return response
}

func (p PriceSubscriptionHandler) TelegramCommand() string {
	return "/s"
}

func (p PriceSubscriptionHandler) ShouldHandle(m telebot.Message) bool {
	return strings.HasPrefix(m.Text, p.TelegramCommand())
}
