package botapi

import (
	"fmt"
	"strings"
	"time"

	"gopkg.in/tucnak/telebot.v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

type Client struct {
	Bot       *telebot.Bot
	Handlers  []MessageHandler
	UpdateID  int
	Log       log.Logger
	Whitelist map[int64]void
}

func NewClient(Token string, opts ...Option) (*Client, error) {
	bot, err := telebot.NewBot(telebot.Settings{Token: Token, Poller: &telebot.LongPoller{Timeout: 60 * time.Second}})
	if err != nil {
		return nil, err
	}

	c := &Client{
		bot,
		nil,
		0,
		&nop.Logger{},
		make(map[int64]void),
	}
	for _, opt := range opts {
		opt.apply(c)
	}
	return c, nil
}

func (c *Client) SendMessage(id int64, text string) {
	//TODO(mikhailche): properly handle error?
	_, err := c.Bot.Send(&telebot.Chat{ID: id}, text)
	if err != nil {
		return
	}
}

func (c *Client) GetMe() User {
	self := c.Bot.Me
	return User{
		ID:           self.ID,
		IsBot:        self.IsBot,
		LanguageCode: self.LanguageCode,
	}
}

func (c *Client) AddHandler(handler MessageHandler) {
	c.Handlers = append(c.Handlers, handler)
}

func (c *Client) PollForever() {
	c.Bot.Handle(telebot.OnText, func(m *telebot.Message) {
		if _, exists := c.Whitelist[m.Chat.ID]; !exists {
			c.Log.Info("Blocked update", log.Int64("Message chat id", m.Chat.ID))
			c.SendMessage(m.Chat.ID, fmt.Sprintf("Chat ID: %d", m.Chat.ID))
			return
		}
		for _, handler := range c.Handlers {
			if handler.ShouldHandle(*m) {
				for response := range handler.Handle(*m) {
					keyboardCommands := make([]string, 0, len(response.Keyboard))
					for _, btn := range response.Keyboard {
						keyboardCommands = append(keyboardCommands, btn.Command)
					}
					c.SendMessage(
						m.Chat.ID,
						strings.Join(
							append([]string{response.Text}, keyboardCommands...),
							"\n",
						),
					)
				}
				return
			}
		}
	})
	c.Bot.Start()
}
