package pricesubimpl

import (
	"context"

	"a.yandex-team.ru/travel/avia/chatbot/public/pricesub"
	result "a.yandex-team.ru/travel/avia/library/proto/search_result/v1"
)

type subscriptionTable struct {
	subscriptions map[int]pricesub.Subscription
	byQkey        map[string][]int
	autoincrement int
}

func newSubscriptionTable() subscriptionTable {
	return subscriptionTable{
		subscriptions: make(map[int]pricesub.Subscription),
		byQkey:        make(map[string][]int),
	}
}

func (s *subscriptionTable) WhereQkey(key pricesub.Query) []*pricesub.Subscription {
	ids := s.byQkey[key.String()]
	var subscriptions []*pricesub.Subscription
	for _, id := range ids {
		sub := s.subscriptions[id]
		subscriptions = append(subscriptions, &sub)
	}
	return subscriptions
}

type subscriberToSubscriptionTable struct {
	subscriberSubscriptions map[int]pricesub.SubscriberSubscription
	bySubscription          map[pricesub.Subscription][]int
	autoincrement           int
}

func (s *subscriberToSubscriptionTable) WhereSubscription(subscription pricesub.Subscription) []pricesub.SubscriberSubscription {
	var ss []pricesub.SubscriberSubscription
	for _, subscriber := range s.bySubscription[subscription] {
		ss = append(ss, s.subscriberSubscriptions[subscriber])
	}
	return ss
}

func newSubscriberToSubscriptionTable() subscriberToSubscriptionTable {
	return subscriberToSubscriptionTable{
		subscriberSubscriptions: make(map[int]pricesub.SubscriberSubscription),
		bySubscription:          make(map[pricesub.Subscription][]int),
	}
}

type subscriberTable struct {
	subscribers []pricesub.Subscriber
}

func newSubscriberTable() subscriberTable {
	return subscriberTable{}
}

type InMemorySubscriptionsRepository struct {
	subscriptionTable
	subscriberToSubscriptionTable
	subscriberTable
}

func (s *subscriberToSubscriptionTable) UpdateLastSeenMinPrice(ctx context.Context, r *result.Result, subsub pricesub.SubscriberSubscription) {
	min := MinPriceVariant(r, subsub.Subscription.Filter)
	subsub.LastSeenMinPrice = min.Price
	s.SaveSubscriberSubscription(ctx, subsub)
}

func (s *subscriberToSubscriptionTable) SaveSubscriberSubscription(ctx context.Context, subsub pricesub.SubscriberSubscription) {
	subscription := subsub.Subscription
	for idx, subscriber := range s.subscriberSubscriptions {
		if subsub.Subscriber.Equals(subscriber.Subscriber) && subsub.Subscription.Query.Equals(subscriber.Subscription.Query) {
			s.subscriberSubscriptions[idx] = subsub
			return
		}
	}
	idx := s.autoincrement
	s.autoincrement += 1

	s.subscriberSubscriptions[idx] = subsub
	s.bySubscription[subscription] = append(s.bySubscription[subscription], idx)
}

func (s *subscriptionTable) findSubscription(qkey string, subscription pricesub.Subscription) *int {
	if idxs, exist := s.byQkey[qkey]; exist {
		for _, idx := range idxs {
			storedSubscription := s.subscriptions[idx]
			if subscription.Filter.Equals(storedSubscription.Filter) {
				return &idx
			}
		}
	}
	return nil
}

func (s *subscriptionTable) SaveSubscription(ctx context.Context, subscription pricesub.Subscription) {
	qkey := subscription.Query.String()
	foundIndex := s.findSubscription(qkey, subscription)
	if foundIndex != nil {
		return
	}
	idx := s.autoincrement
	s.autoincrement++
	s.subscriptions[idx] = subscription
	s.byQkey[qkey] = append(s.byQkey[qkey], idx)
}

func (s *subscriberTable) SaveSubscriber(ctx context.Context, subscriber pricesub.Subscriber) {
	for _, s := range s.subscribers {
		if s == subscriber {
			return
		}
	}
	s.subscribers = append(s.subscribers, subscriber)
}

func NewInMemoryRepository() *InMemorySubscriptionsRepository {
	return &InMemorySubscriptionsRepository{
		subscriptionTable:             newSubscriptionTable(),
		subscriberToSubscriptionTable: newSubscriberToSubscriptionTable(),
		subscriberTable:               newSubscriberTable(),
	}
}

func (s *InMemorySubscriptionsRepository) FindSubscriptionsForResult(result *result.Result) []*pricesub.Subscription {
	qkey := constructQKey(result)
	subscriptions := s.subscriptionTable.WhereQkey(qkey)
	return subscriptions
}

func constructQKey(r *result.Result) pricesub.Query {
	return pricesub.Query{
		PointFrom:       r.PointFrom,
		PointTo:         r.PointTo,
		When:            r.DateForward,
		Return:          r.DateBackward,
		Passengers:      r.Passengers,
		ServiceClass:    r.ServiceClass,
		NationalVersion: r.NationalVersion,
	}
}

func (s *InMemorySubscriptionsRepository) FindSubscriberSubscriptions(subscription pricesub.Subscription) []pricesub.SubscriberSubscription {
	return s.subscriberToSubscriptionTable.WhereSubscription(subscription)
}
