package pricesubimpl

import (
	"fmt"

	"gopkg.in/tucnak/telebot.v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/chatbot/public/pricesub"
	searchResult "a.yandex-team.ru/travel/avia/chatbot/public/search_result"
	"a.yandex-team.ru/travel/avia/library/proto/common/v1"
	result "a.yandex-team.ru/travel/avia/library/proto/search_result/v1"
)

type TelegramSender struct {
	Bot    *telebot.Bot
	Logger log.Logger
}

func (s *TelegramSender) Send(r *result.Result, subscription *pricesub.SubscriberSubscription) {
	var min = MinPriceVariant(r, subscription.Subscription.Filter)

	charter := ""
	if min.GetCharter() {
		charter = "чартер"
	}
	variantURL := searchResult.NewVariantURL(r, min)
	chat := &telebot.Chat{ID: subscription.Subscriber.ChatID}
	message := fmt.Sprintf(
		"Минимальная цена для направления %s.\n"+
			"%.2f %s %s от партнёра %s %s\n",
		subscription.Subscription.Query.String(),
		min.GetPrice().GetValue(), min.GetPrice().GetCurrency(), charter, min.PartnerCode, variantURL.TinyURL(variantURL.OrderURL()),
	)
	if min.SelfConnect {
		message += "Осторожно: этот вариант с самостоятельной пересадкой.\n"
	}
	message += s.transfersInfo(min)
	message += s.baggageInfo(min)

	msg, err := s.Bot.Send(chat, message)
	if err != nil {
		s.Logger.Error("Cannot send message", log.Any("msg", msg), log.Error(err))
	}
}

func (s *TelegramSender) transfersInfo(min *result.Variant) string {
	if len(min.Forward) > 1 || len(min.Backward) > 1 {
		return ""
	}
	return "Без пересадок.\n"
}

func (s *TelegramSender) baggageInfo(min *result.Variant) string {
	var allSegmentsBaggageIncluded, someSegmentsBaggageIncluded bool

	allSegmentsBaggageIncluded = true
	someSegmentsBaggageIncluded = false
	for _, segment := range append(append([]*result.FlightSegment{}, min.Forward...), min.Backward...) {
		if segment.Baggage.Included {
			someSegmentsBaggageIncluded = true
		} else {
			allSegmentsBaggageIncluded = false
		}
	}
	if allSegmentsBaggageIncluded {
		return "Багаж включен в стоимость билета.\n"
	} else {
		if someSegmentsBaggageIncluded {
			return "Багаж не включен в стоимость для всех сегментов перелёта.\n"
		} else {
			return "Без багажа.\n"
		}
	}
}

func (s *TelegramSender) ShouldSend(r *result.Result, subscription *pricesub.SubscriberSubscription) bool {
	variant := MinPriceVariant(r, subscription.Subscription.Filter)
	if variant == nil {
		return false
	}
	return SmartDiffers(variant.Price, subscription.LastSeenMinPrice)
}

func SmartDiffers(price1 *common.Price, price2 *common.Price) bool {
	if (price1 == nil) != (price2 == nil) {
		return true
	}
	if price1 == nil {
		return false
	}
	val1, val2 := price1.Value, price2.Value
	absDiff := val1 - val2
	if absDiff < 0 {
		absDiff = -absDiff
	}
	relDiff1 := absDiff / val1
	relDiff2 := absDiff / val2
	var relDiff float64
	if relDiff1 > relDiff2 {
		relDiff = relDiff1
	} else {
		relDiff = relDiff2
	}
	return relDiff > .05 && absDiff > 100
}

func MinPriceVariant(r *result.Result, filter pricesub.Filter) *result.Variant {
	var minPriceVariant *result.Variant = nil
nextVariant:
	for _, variant := range r.GetVariants() {
		if filter.Baggage != nil {
			for _, segment := range append(append([]*result.FlightSegment{}, variant.Forward...), variant.Backward...) {
				if !*filter.Baggage {
					continue
				}
				if segment.Baggage.Included || segment.Baggage.Pieces > 0 || segment.Baggage.Weight > 0 {
					continue
				}
				continue nextVariant
			}
		}
		if filter.MaxTransferCount != nil {
			forwardTransfers := len(variant.Forward) - 1
			backwardTransfers := len(variant.Backward) - 1
			if forwardTransfers > *filter.MaxTransferCount || backwardTransfers > *filter.MaxTransferCount {
				continue nextVariant
			}
		}
		if minPriceVariant == nil ||
			(minPriceVariant.Price.Currency == variant.Price.Currency &&
				minPriceVariant.Price.Value > variant.Price.Value) {
			minPriceVariant = variant
		}
	}
	return minPriceVariant
}
