package pricesub

import (
	"fmt"
	"strconv"
	"strings"

	searchresult "a.yandex-team.ru/travel/avia/chatbot/public/search_result"
	"a.yandex-team.ru/travel/avia/library/proto/common/v1"
	travel_commons_proto "a.yandex-team.ru/travel/proto"
)

type Subscriber struct {
	ChatID int64
}

func (s Subscriber) Equals(subscriber Subscriber) bool {
	return s.ChatID == subscriber.ChatID
}

type SubscriberSubscription struct {
	Subscriber       Subscriber
	Subscription     Subscription
	LastSeenMinPrice *common.Price
}

type Subscription struct {
	Query  Query
	Filter Filter
}

func (s Subscription) Equals(subscription Subscription) bool {
	return s.Query.Equals(subscription.Query) && s.Filter.Equals(subscription.Filter)
}

type Query struct {
	PointFrom       *common.Point
	PointTo         *common.Point
	When            *travel_commons_proto.TDate
	Return          *travel_commons_proto.TDate
	Passengers      *common.Passengers
	ServiceClass    common.ServiceClass
	NationalVersion common.NationalVersion
}

func (q Query) String() string {
	from := searchresult.ConvertPointToString(q.PointFrom)
	to := searchresult.ConvertPointToString(q.PointTo)

	fnTDateToString := func(d *travel_commons_proto.TDate) string {
		if d == nil {
			return "None"
		}
		return fmt.Sprintf("%04d-%02d-%02d", d.Year, d.Month, d.Day)
	}
	when := fnTDateToString(q.When)
	backward := fnTDateToString(q.Return)
	adults := strconv.Itoa(int(q.Passengers.Adults))
	children := strconv.Itoa(int(q.Passengers.Children))
	infants := strconv.Itoa(int(q.Passengers.Infants))
	service := searchresult.ServiceClassToQkeyString(q.ServiceClass)
	nv := searchresult.NationalVersionToString(q.NationalVersion)
	return strings.Join([]string{from, to, when, backward, adults, children, infants, service, nv}, "_")
}

func (q Query) Equals(query Query) bool {
	return q.PointFrom.Id == query.PointFrom.Id &&
		q.PointFrom.Type == query.PointFrom.Type &&

		q.PointTo.Id == query.PointTo.Id &&
		q.PointTo.Type == query.PointTo.Type &&

		q.When.Year == query.When.Year &&
		q.When.Month == query.When.Month &&
		q.When.Day == query.When.Day &&

		((q.Return == nil) == (query.Return == nil) && // both nil or both not nil
			(q.Return == nil || // either nil or compare values
				(q.Return.Year == query.Return.Year &&
					q.Return.Month == query.Return.Month &&
					q.Return.Day == query.Return.Day))) &&

		q.Passengers.Adults == query.Passengers.Adults &&
		q.Passengers.Children == query.Passengers.Children &&
		q.Passengers.Infants == query.Passengers.Infants &&

		q.ServiceClass == query.ServiceClass &&
		q.NationalVersion == query.NationalVersion
}

func QueryFromQkey(qkey string) (*Query, error) {
	parts := strings.Split(qkey, "_")
	from, to, when, backward, adults, children, infants, service, nv := parts[0], parts[1], parts[2], parts[3], parts[4], parts[5], parts[6], parts[7], parts[8]

	pointFrom, err := searchresult.ConvertStringToPoint(from)
	if err != nil {
		return nil, err
	}
	pointTo, err := searchresult.ConvertStringToPoint(to)
	if err != nil {
		return nil, err
	}
	dateForward, err := TDateFromString(when)
	if err != nil {
		return nil, err
	}
	dateBackward, err := TDateFromString(backward)
	if err != nil {
		return nil, err
	}
	passengers, err := PassengersFromString(adults, children, infants)
	if err != nil {
		return nil, err
	}
	serviceClass := ServiceClassFromString(service)
	nationalVersion := NationalVersionFromString(nv)
	return &Query{
		PointFrom:       pointFrom,
		PointTo:         pointTo,
		When:            dateForward,
		Return:          dateBackward,
		Passengers:      passengers,
		ServiceClass:    serviceClass,
		NationalVersion: nationalVersion,
	}, nil
}

func TDateFromString(backward string) (*travel_commons_proto.TDate, error) {
	if backward == "None" {
		return nil, nil
	}
	parts := strings.Split(backward, "-")
	SYear, SMonth, SDay := parts[0], parts[1], parts[2]
	Year, err := strconv.Atoi(SYear)
	if err != nil {
		return nil, err
	}
	Month, err := strconv.Atoi(SMonth)
	if err != nil {
		return nil, err
	}
	Day, err := strconv.Atoi(SDay)
	if err != nil {
		return nil, err
	}
	return &travel_commons_proto.TDate{
		Year:  int32(Year),
		Month: int32(Month),
		Day:   int32(Day),
	}, nil
}

func PassengersFromString(Sadults string, Schildren string, Sinfants string) (*common.Passengers, error) {
	adults, err := strconv.Atoi(Sadults)
	if err != nil {
		return nil, err
	}
	children, err := strconv.Atoi(Schildren)
	if err != nil {
		return nil, err
	}
	infants, err := strconv.Atoi(Sinfants)
	if err != nil {
		return nil, err
	}
	return &common.Passengers{
		Adults:   uint32(adults),
		Children: uint32(children),
		Infants:  uint32(infants),
	}, nil
}

func ServiceClassFromString(service string) common.ServiceClass {
	return common.ServiceClass(common.ServiceClass_value[service])
}

func NationalVersionFromString(nv string) common.NationalVersion {
	return common.NationalVersion(common.NationalVersion_value[nv])
}

type Filter struct {
	Baggage          *bool
	MaxTransferCount *int
}

func (f Filter) Equals(filter Filter) bool {
	return (f.Baggage == filter.Baggage || (f.Baggage != nil && filter.Baggage != nil && *f.Baggage == *filter.Baggage)) &&
		(f.MaxTransferCount == filter.MaxTransferCount || (f.MaxTransferCount != nil && filter.MaxTransferCount != nil && *f.MaxTransferCount == *filter.MaxTransferCount))
}
