package searchresult

import (
	"encoding/json"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/library/proto/common/v1"
	result "a.yandex-team.ru/travel/avia/library/proto/search_result/v1"
	travel_commons_proto "a.yandex-team.ru/travel/proto"
)

const orderPath = "avia/order/"
const TravelHostTesting = "travel-test.yandex.ru"
const TravelHostProduction = "travel.yandex.ru"

func NewVariantURL(r *result.Result, v *result.Variant, options ...func(*VariantURL)) *VariantURL {
	u := &VariantURL{
		Host:         TravelHostProduction,
		PointFrom:    r.GetPointFrom(),
		PointTo:      r.GetPointTo(),
		When:         r.GetDateForward(),
		ReturnDate:   r.GetDateBackward(),
		Passengers:   r.GetPassengers(),
		ServiceClass: r.GetServiceClass(),
		Variant:      v,
		Flights:      r.GetFlights(),
	}
	for _, option := range options {
		option(u)
	}
	return u
}

func WithTestingHost(v *VariantURL) {
	v.Host = TravelHostTesting
}

type VariantURL struct {
	Host         string
	PointFrom    *common.Point
	PointTo      *common.Point
	When         *travel_commons_proto.TDate
	ReturnDate   *travel_commons_proto.TDate
	Passengers   *common.Passengers
	ServiceClass common.ServiceClass
	Variant      *result.Variant
	Flights      map[string]*result.Flight
}

func (r *VariantURL) TinyURL(inputURL string) string {
	response, err := http.Get((&url.URL{
		Scheme:   "https",
		Host:     r.Host,
		Path:     "api/common/storeTinyUrl",
		RawQuery: url.Values{"url": []string{inputURL}}.Encode(),
	}).String())
	if err != nil {
		return inputURL
	}
	var tinyURL string
	err = json.NewDecoder(response.Body).Decode(&tinyURL)
	if err != nil {
		return inputURL
	}
	return tinyURL
}

func (r *VariantURL) OrderURL() string {
	// https://travel-test.yandex.ru/avia/order/?adult_seats=1&backward=S7%202042.2021-09-08T09%3A30%2CS7%201175.2021-09-08T14%3A55&children_seats=0&forward=S7%201176.2021-09-01T20%3A05%2CS7%202051.2021-09-01T23%3A55&fromId=c54&infant_seats=0&klass=economy&oneway=2&return_date=2021-09-08&toId=c239&when=2021-09-01#empty
	q := url.Values{
		"fromId":         {r.getPointFromID()},
		"toId":           {r.getPointToID()},
		"when":           {r.getWhen()},
		"return_date":    {r.getReturnDate()},
		"adult_seats":    {r.getAdultSeats()},
		"children_seats": {r.getChildrenSeats()},
		"infant_seats":   {r.getInfantSeats()},
		"klass":          {r.getClass()},
		"oneway":         {r.getOneway()},
		"forward":        {r.getForwardPath()},
		"backward":       {r.getBackwardPath()},
		"baggage":        r.getBaggage(),
	}
	fragment := "empty"
	if q["baggage"] != nil {
		fragment = "bg=1"
	}
	return (&url.URL{
		Scheme:     "https",
		Host:       r.Host,
		Path:       orderPath,
		ForceQuery: false,
		RawQuery:   q.Encode(),
		Fragment:   fragment,
	}).String()
}
func (r *VariantURL) getPointFromID() string {
	return ConvertPointToString(r.PointFrom)
}
func (r *VariantURL) getPointToID() string {
	return ConvertPointToString(r.PointTo)
}

func ConvertPointToString(p *common.Point) string {
	if p == nil {
		return "nil"
	}
	prefix := "c"
	switch p.Type {
	case common.PointType_POINT_TYPE_SETTLEMENT:
		prefix = "c"
	case common.PointType_POINT_TYPE_STATION:
		prefix = "s"
	case common.PointType_POINT_TYPE_REGION:
		prefix = "r"
	case common.PointType_POINT_TYPE_COUNTRY:
		prefix = "l"
	}
	return prefix + strconv.FormatUint(p.Id, 10)
}

func ConvertStringToPoint(s string) (*common.Point, error) {
	rr := []rune(s)
	prefix := rr[0]
	var t common.PointType
	switch prefix {
	case 'c':
		t = common.PointType_POINT_TYPE_SETTLEMENT
	case 's':
		t = common.PointType_POINT_TYPE_STATION
	case 'r':
		t = common.PointType_POINT_TYPE_REGION
	case 'l':
		t = common.PointType_POINT_TYPE_COUNTRY
	default:
		return nil, xerrors.Errorf("unknown point type %s in point %s", string(prefix), s)
	}
	id, err := strconv.Atoi(string(rr[1:]))
	if err != nil {
		return nil, err
	}
	return &common.Point{
		Type: t,
		Id:   uint64(id),
	}, nil
}

func (r *VariantURL) getWhen() string {
	return convertTDateToDateString(r.When)
}

func (r *VariantURL) getReturnDate() string {
	return convertTDateToDateString(r.ReturnDate)
}

func convertTDateToDateString(date *travel_commons_proto.TDate) string {
	if date == nil {
		return ""
	}
	return time.Date(int(date.GetYear()), time.Month(date.GetMonth()), int(date.GetDay()), 0, 0, 0, 0, time.UTC).
		Format("2006-01-02")
}

func (r *VariantURL) getAdultSeats() string {
	return strconv.FormatUint(uint64(r.Passengers.GetAdults()), 10)
}

func (r *VariantURL) getChildrenSeats() string {
	return strconv.FormatUint(uint64(r.Passengers.GetChildren()), 10)
}

func (r *VariantURL) getInfantSeats() string {
	return strconv.FormatUint(uint64(r.Passengers.GetInfants()), 10)
}

func (r *VariantURL) getClass() string {
	return ServiceClassToQkeyString(r.ServiceClass)
}

func (r *VariantURL) getOneway() string {
	if r.ReturnDate == nil {
		return "1"
	}
	return "2"
}

func (r *VariantURL) getForwardPath() string {
	return getPath(r.Variant.Forward, r.Flights)
}
func (r *VariantURL) getBackwardPath() string {
	return getPath(r.Variant.Backward, r.Flights)
}

func (r *VariantURL) getBaggage() []string {
	someBaggage := false
	for _, s := range r.Variant.Forward {
		someBaggage = someBaggage || s.GetBaggage().GetIncluded()
	}
	for _, s := range r.Variant.Backward {
		someBaggage = someBaggage || s.GetBaggage().GetIncluded()
	}
	if someBaggage {
		return []string{"1"}
	}
	return nil
}

func getPath(ss []*result.FlightSegment, flights map[string]*result.Flight) string {
	//"S7 1176.2021-09-01T20:05,S7 2051.2021-09-01T23:55"
	var segments []string
	for _, s := range ss {
		segments = append(segments, getPathSegment(s, flights))
	}
	return strings.Join(segments, ",")
}

func getPathSegment(s *result.FlightSegment, flights map[string]*result.Flight) string {
	//"S7 1176.2021-09-01T20:05"
	flight := flights[s.GetFlightKey()]
	withSeconds := flight.GetLocalDeparture()
	strippedSeconds := withSeconds[:strings.LastIndex(withSeconds, ":")]
	return flight.GetNumber() + "." + strippedSeconds
}
