package sharedflights

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
)

const APIHost = "shared-flights.production.avia.yandex.net"

type Client struct {
	Host string
	HTTP http.Client
}

func (c *Client) Flight(carrier, flightNumber, date string) FlightSegment {

	resp, err := c.HTTP.Get(
		fmt.Sprintf(
			"http://%v/api/v1/flight/%v/%v/%v/",
			url.QueryEscape(c.Host), url.QueryEscape(carrier), url.QueryEscape(flightNumber), url.QueryEscape(date),
		),
	)
	if err != nil {
		return FlightSegment{}
	}
	fs := FlightSegment{}
	_ = json.NewDecoder(resp.Body).Decode(&fs)
	return fs
}

func (c *Client) FlightNumbersToCarriers(flightNumbers []string) map[string]*int64 {
	URL := url.URL{
		Scheme:   "http",
		Host:     c.Host,
		Path:     "api/v1/flight-numbers-to-carriers/",
		RawQuery: url.Values{"flight_numbers": flightNumbers}.Encode(),
	}
	resp, err := c.HTTP.Get(URL.String())
	if err != nil {
		return nil
	}
	var response map[string]*int64
	_ = json.NewDecoder(resp.Body).Decode(&response)
	return response
}

// This is what's printed into response
type FlightSegment struct {
	CompanyIata       string           `json:"airlineCode"`
	CompanyRaspID     int32            `json:"airlineID"`
	Number            string           `json:"number"`
	AirportFromIata   string           `json:"airportFromCode"`
	AirportFromRaspID int64            `json:"airportFromID"`
	DepartureDay      string           `json:"departureDay"`
	DepartureTime     string           `json:"departureTime"`
	DepartureTzName   string           `json:"departureTimezone"`
	DepartureUTC      string           `json:"departureUtc"`
	DepartureTerminal string           `json:"departureTerminal"`
	AirportToIata     string           `json:"airportToCode"`
	AirportToRaspID   int64            `json:"airportToID"`
	ArrivalDay        string           `json:"arrivalDay"`
	ArrivalTime       string           `json:"arrivalTime"`
	ArrivalTzName     string           `json:"arrivalTimezone"`
	ArrivalUTC        string           `json:"arrivalUtc"`
	ArrivalTerminal   string           `json:"arrivalTerminal"`
	CreatedAtUTC      string           `json:"createdAtUtc"`
	UpdatedAtUTC      string           `json:"updatedAtUtc"`
	Status            FlightStatus     `json:"status,omitempty"`
	Segments          []*FlightSegment `json:"segments,omitempty"`
	Source            string           `json:"source"`
	Title             string           `json:"title,omitempty"`
	Banned            string           `json:"banned,omitempty"`
	TransportModelID  int64            `json:"transportModelID"`
}

type FlightStatus struct {
	Status                FlightStatusText `json:"status"`
	DepartureDT           string           `json:"departure"`
	ArrivalDT             string           `json:"arrival"`
	DepartureStatus       FlightStatusText `json:"departureStatus"`
	ArrivalStatus         FlightStatusText `json:"arrivalStatus"`
	ArrivalGate           string           `json:"arrivalGate"`
	ArrivalTerminal       string           `json:"arrivalTerminal"`
	DepartureGate         string           `json:"departureGate"`
	DepartureTerminal     string           `json:"departureTerminal"`
	DepartureSource       string           `json:"departureSource"`
	ArrivalSource         string           `json:"arrivalSource"`
	CheckInDesks          string           `json:"checkInDesks"`
	BaggageCarousels      string           `json:"baggageCarousels"`
	CreatedAtUTC          string           `json:"createdAtUtc"`
	UpdatedAtUTC          string           `json:"updatedAtUtc"`
	DepartureUpdatedAtUTC string           `json:"departureUpdatedAtUtc"`
	ArrivalUpdatedAtUTC   string           `json:"arrivalUpdatedAtUtc"`
	Diverted              bool             `json:"diverted"`
	DivertedAirportCode   string           `json:"divertedAirportCode"`
	DivertedAirportID     int32            `json:"divertedAirportID"`
}

type FlightStatusText string

const (
	FlightStatusUnknown   FlightStatusText = "unknown"
	FlightStatusCancelled FlightStatusText = "cancelled"

	// generic status means that it's either departed from departure airport (FlightStatusDeparted) or arrived to arrival airport (FlightStatusArrived)
	FlightStatusFinal    FlightStatusText = "final"
	FlightStatusDeparted FlightStatusText = "departed"
	FlightStatusArrived  FlightStatusText = "arrived"

	FlightStatusDelayed FlightStatusText = "delayed"
	FlightStatusEarly   FlightStatusText = "early"
	FlightStatusOnTime  FlightStatusText = "on_time"
)
