import copy
import logging
import traceback
from pydantic import parse_obj_as
from typing import List
import yt.wrapper as yt

from travel.avia.country_restrictions.lib.parsers.to_yt_table_parser import ToYtTableParser
from travel.avia.country_restrictions.lib.table_format.base_format import BaseFormat
from travel.avia.country_restrictions.lib.table_format.metrics_as_columns_format import MetricsAsColumnsFormat
from travel.avia.country_restrictions.lib.types import Metric
from travel.avia.country_restrictions.lib.types.metric_type import ALL_METRICS
from travel.avia.country_restrictions.parsers import get_parsers_tables_by_id_dict, PARSERS_PRIORITY


class CombinerParser(ToYtTableParser):
    METRIC_TYPES = ALL_METRICS

    UPDATING_TABLE_NAME = 'combiner-as-columns'
    OUTPUT_TABLE_FORMATS: List[BaseFormat] = [
        MetricsAsColumnsFormat(output_table_short_name='combiner-as-columns'),
    ]

    PARSER_NAME = 'combiner'
    SKIP_PREVIOUS_DATA = True

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.input_sources = get_parsers_tables_by_id_dict(**kwargs)
        self.parsers_priority = PARSERS_PRIORITY

    def get_data_from_sources(self):
        result = {}
        for source_id, table_name in self.input_sources.items():
            try:
                table = [dict(row) for row in self.yt_client.read_table(yt.TablePath(table_name))]
                result[source_id] = table
            except yt.errors.YtResolveError:
                logging.getLogger().exception(f'Unable to get data from yt: {traceback.format_exc()}')
                continue

        return result

    def get_data(self, old_data):
        data = {}
        source_data = self.get_data_from_sources()

        for source in self.parsers_priority:
            table = copy.deepcopy(source_data.get(source.source, None))
            if table is None:
                continue

            for row in table:
                key = self.get_point_key_from_row(row)
                if source.geo is not None and key not in source.geo:
                    continue

                if key not in data:
                    data[key] = {}

                for column_name, column_value in dict(row).items():
                    if source.metrics is not None and column_name not in source.metrics:
                        continue

                    exists_in_metrics = False
                    for metric in self.METRIC_TYPES:
                        if metric.name == column_name:
                            exists_in_metrics = True
                            break

                    if column_name not in data[key] and column_value is not None and exists_in_metrics:
                        metric = parse_obj_as(Metric, column_value)
                        data[key][column_name] = metric
                        metric.source = source.source
        return data
