from pydantic import parse_obj_as
from typing import List
import yt.wrapper as yt

from travel.avia.country_restrictions.aggregator.hierarchy_applier.apply_hierarchy import build_tree, Stats
from travel.avia.country_restrictions.lib.geo_format_manager import GeoFormatManager
from travel.avia.country_restrictions.lib.parsers.to_yt_table_parser import ToYtTableParser
from travel.avia.country_restrictions.lib.table_format.base_format import BaseFormat
from travel.avia.country_restrictions.lib.table_format.metrics_as_columns_format import MetricsAsColumnsFormat
from travel.avia.country_restrictions.lib.types import CountryInfo, InformationTable
from travel.avia.country_restrictions.lib.types.metric_type import ALL_METRICS


class HierarchyApplierParser(ToYtTableParser):
    METRIC_TYPES = ALL_METRICS

    INPUT_TABLE_NAME = 'combiner-as-columns'
    UPDATING_TABLE_NAME = 'apply-hierarchy-as-columns'
    OUTPUT_TABLE_FORMATS: List[BaseFormat] = [
        MetricsAsColumnsFormat(output_table_short_name='apply-hierarchy-as-columns'),
    ]

    PARSER_NAME = 'hierarchy-applier'
    SKIP_PREVIOUS_DATA = True

    def get_input_table_fullname(self) -> str:
        return self.get_path_by_shortname(self.INPUT_TABLE_NAME)

    def get_data(self, old_data):
        data: InformationTable = {}
        source = [dict(row) for row in self.yt_client.read_table(yt.TablePath(self.get_input_table_fullname()))]

        for row in source:
            key = self.get_point_key_from_row(row)
            if key is None:
                continue

            data[key] = parse_obj_as(CountryInfo, row)

            for metric_name, metric in data[key].items():
                if metric is not None:
                    metric.point_key = key

        return self.apply_hierarchy(data, self.geo_format_manager)

    def apply_hierarchy(self, info_table: InformationTable, geo_format_manager: GeoFormatManager) -> InformationTable:
        # Algorithm of making hierarchy
        # 1. Building parent-child tree for all geo points
        # 2. Performing depth-first search for the tree forwarding metrics from parent to children.

        root_node, tree_node_storage = build_tree(info_table, geo_format_manager)
        stats = Stats()
        root_node.update_metrics_with_hierarchy({}, stats)
        self.solomon_pusher.push(
            'stats',
            stats.replaced_cells,
            {'stats_data_type': 'cells', 'stats_state_type': 'replaced'},
        )

        return {point_key: tree_node.country_info for point_key, tree_node in tree_node_storage.items()}
