import logging
from typing import List

import click
from pydantic import parse_obj_as
import requests
from tvmauth import TvmApiClientSettings, TvmClient, TvmClientStatus

from travel.avia.country_restrictions.aggregator import AggregatorParser
from travel.avia.country_restrictions.combined import CombinedParser
from travel.avia.country_restrictions.lib.parsers.to_yt_table_parser import ToYtTableParser
from travel.avia.country_restrictions.lib.types import Environment
from travel.avia.country_restrictions.lib.kwargs_prepare import kwargs_prepare
from travel.avia.country_restrictions.lib.utils.multi_cluster_yt_client import MultiClusterYtClient
from travel.avia.country_restrictions.parsers import get_parsers_class_by_id_dict
from travel.avia.country_restrictions.lib.types.rich_string import RichString

YT_RESULTS_FOLDER_FORMAT = 'https://yt.yandex-team.ru/{}/navigation?path={}'


def print_result_yt_folders(function):
    def internal(**kwargs):
        function(**kwargs)

        logging.getLogger().error('Work is done. Check your results here:')
        for proxy in kwargs['yt_proxies']:
            yt_path = ToYtTableParser.get_base_path_static(kwargs['base_path'], kwargs['environment'], kwargs['version'])
            logging.getLogger().error(YT_RESULTS_FOLDER_FORMAT.format(proxy, yt_path))

    return internal


@click.group('country restrictions')
def group():
    pass


def yt_proxies_callback(ctx, param, value):
    if value is not None:
        return value.split(',')


def environment_callback(ctx, param, value):
    if value is not None:
        if isinstance(value, str):
            return Environment.from_str(value)
        return value


def cli_args(function):
    env_dec = click.option(
        '--environment',
        'environment',
        default=Environment.TESTING.value,
        show_default=True,
        type=click.Choice([x.value for x in Environment]),
        help='environment folder',
        callback=environment_callback,
    )
    version_dec = click.option('--version', 'version', default=1, show_default=True, type=int, help='version folder')
    base_path_dec = click.option(
        '--base-path', 'base_path', type=str, required=True, prompt=True, help='YT directory for project root'
    )
    solomon_token_dec = click.option(
        '--solomon-token',
        'solomon_token',
        type=str,
        required=False,
        help='token for solomon',
        default=None,
    )
    sandbox_token_dec = click.option(
        '--sandbox-token',
        'sandbox_token',
        type=str,
        required=False,
        help='token for sandbox',
        default=None,
    )
    yt_token_dec = click.option('--yt-token', 'yt_token', type=str, required=False, help='token for YT')
    yt_proxies_dec = click.option(
        '--yt_proxies',
        'yt_proxies',
        type=str,
        required=True,
        default=','.join(ToYtTableParser.YT_PROXIES),
        show_default=True,
        help='token for YT',
        callback=yt_proxies_callback,
    )

    return base_path_dec(env_dec(version_dec(solomon_token_dec(sandbox_token_dec(yt_token_dec(yt_proxies_dec(function)))))))


@group.command('source-parser')
@click.argument('parser_name')
@cli_args
@print_result_yt_folders
def source_parser(parser_name: str, **kwargs):
    parser_class = get_parsers_class_by_id_dict()[parser_name]
    parser_class(**kwargs_prepare(**kwargs)).run()


@group.command('aggregator')
@cli_args
@print_result_yt_folders
def aggregator(**kwargs):
    AggregatorParser(**kwargs_prepare(**kwargs)).run()


@group.command('combined')
@cli_args
@print_result_yt_folders
def combined(**kwargs):
    CombinedParser(**kwargs_prepare(**kwargs)).run()


@group.command('yt-clear')
@cli_args
@print_result_yt_folders
def yt_clear(**kwargs):
    base_path = kwargs['base_path']
    yt_proxies = kwargs['yt_proxies']
    yt_token = kwargs.get('yt_token', None)

    if 'dev' not in base_path:
        raise Exception('This program does not remove non-developer\'s YT directory, do it by hands!')

    yt_client = MultiClusterYtClient(proxies=yt_proxies, token=yt_token)
    yt_path = ToYtTableParser.get_base_path_static(**kwargs)
    logging.getLogger().warning(f'Removing {yt_path}')
    yt_client.remove(yt_path, recursive=True)
    yt_client.create('map_node', yt_path, recursive=True, ignore_existing=True)


@group.command('yt-clone')
@cli_args
@print_result_yt_folders
def yt_clone(base_path, yt_proxies, yt_token=None, **kwargs):
    if 'dev' not in base_path:
        raise Exception('This program does not remove non-developer\'s YT directory, do it by hands!')

    yt_client = MultiClusterYtClient(proxies=yt_proxies, token=yt_token)
    yt_client.remove(base_path, recursive=True)
    yt_client.copy('//home/avia/data/country-restrictions', base_path, recursive=True, ignore_existing=True)


@group.command('create-bp')
@cli_args
@print_result_yt_folders
def create_bp(base_path: str, yt_proxies: List[str], yt_token, environment: Environment, version: int, **kwargs):
    """
    Creates base path
    """

    if 'dev' not in base_path:
        raise Exception('This program does not affect non-developer\'s YT directory, do it by hands!')

    yt_client = MultiClusterYtClient(proxies=yt_proxies, token=yt_token)
    yt_client.create(
        'map_node',
        ToYtTableParser.get_base_path_static(base_path, environment, version),
        recursive=True,
        ignore_existing=True,
    )


@group.command('api-req')
@click.option('--pk', 'pk', type=str)
@click.option('--tvm', 'tvm', type=str)
@click.option('--host', 'host', type=str, default='api.travel-balancer-test.yandex.net')
def travel_api_request(pk: str, tvm: str, host: str):
    cli_tvm_service_id = 2002574
    api_tvm_service_id = 2002548
    settings = TvmApiClientSettings(
        self_tvm_id=cli_tvm_service_id,
        enable_service_ticket_checking=True,
        enable_user_ticket_checking=False,
        self_secret=tvm,
        dsts={'api': api_tvm_service_id},
    )
    client = TvmClient(settings)
    if client.status != TvmClientStatus.Ok:
        raise Exception("tvm client has bad status: " + TvmClient.status_to_string(client.status))
    ticket = client.get_service_ticket_for("api")
    r = requests.get(
        f'http://{host}/api/avia_country_restrictions/v1/data?toPointKey={pk}',
        headers={'X-Ya-Service-Ticket': ticket}
    )
    metrics = r.json()['destination']['metrics']
    for metric_name, metric_data in metrics.items():
        logging.getLogger().warning(metric_name)
        logging.getLogger().warning('--------------------------------------------------------------------------------')
        rpi = metric_data['return_point_info']['point_key']
        logging.getLogger().warning(f'Info for point: {rpi}')
        base_value = metric_data['value']
        logging.getLogger().warning(f'Base value:     {base_value}')
        text = parse_obj_as(RichString, metric_data['text']).get_string_description()
        logging.getLogger().warning(f'Text:           {text}')

        additions = parse_obj_as(List[RichString], metric_data['additions'])
        for i in range(len(additions)):
            data = additions[i].get_string_description()
            if i == 0:
                logging.getLogger().warning(f'Additions:      {data}')
            else:
                logging.getLogger().warning(f'                {data}')

        exclusions = parse_obj_as(List[RichString], metric_data['exclusions'])
        for i in range(len(exclusions)):
            data = exclusions[i].get_string_description()
            if i == 0:
                logging.getLogger().warning(f'Exclusions:     {data}')
            else:
                logging.getLogger().warning(f'                {data}')
        logging.getLogger().warning('================================================================================')
        logging.getLogger().warning('\n')


def main():
    group()


if __name__ == '__main__':
    main()
