import datetime
from typing import List

from travel.avia.country_restrictions.lib.types.metric_type import TOURISM_AVAILABILITY, \
    TOURISM_AVAILABLE_FROM_METRIC_TYPE, APPROVED_VACCINES, IS_SPUTNIK_APPROVED
from travel.avia.country_restrictions.lib.types import CountryInfo


metric_correctness_validators = []


def metric_correctness_validator(func):
    metric_correctness_validators.append(func)
    return func


@metric_correctness_validator
def false_tourism_true_vaccines(country_info: CountryInfo) -> List[str]:
    tourism_metric = country_info.get(TOURISM_AVAILABILITY.name, None)
    vaccines_metric = country_info.get(APPROVED_VACCINES.name, None)
    if tourism_metric is None or vaccines_metric is None:
        return []

    if tourism_metric.value is False and len(vaccines_metric.value) > 0:
        return ['false_tourism_true_vaccines']
    else:
        return []


@metric_correctness_validator
def false_tourism_true_sputnik(country_info: CountryInfo) -> List[str]:
    tourism_metric = country_info.get(TOURISM_AVAILABILITY.name, None)
    sputnik_metric = country_info.get(IS_SPUTNIK_APPROVED.name, None)
    if tourism_metric is None or sputnik_metric is None:
        return []

    if tourism_metric.value is False and sputnik_metric.value is True:
        return ['false_tourism_true_sputnik']
    else:
        return []


@metric_correctness_validator
def sputnik_in_vaccines_not_equal(country_info: CountryInfo) -> List[str]:
    vaccines_metric = country_info.get(APPROVED_VACCINES.name, None)
    sputnik_metric = country_info.get(IS_SPUTNIK_APPROVED.name, None)
    if vaccines_metric is None or sputnik_metric is None or sputnik_metric.value is None:
        return []

    sputnik_exists_in_vaccine_list = False

    for vaccine in vaccines_metric.value:
        for elem in ('спутник', 'sputnik'):
            if elem in vaccine:
                sputnik_exists_in_vaccine_list = True
                break

    if sputnik_exists_in_vaccine_list != sputnik_metric.value:
        if sputnik_exists_in_vaccine_list:
            return ['false_sputnik_true_sputnik_in_vaccines_list']
        else:
            return ['true_sputnik_false_sputnik_in_vaccines_list']
    else:
        return []


@metric_correctness_validator
def country_is_closed_but_opened_date_is_reached(country_info: CountryInfo) -> List[str]:
    tourism_availability = country_info.get(TOURISM_AVAILABILITY.name, None)
    tourism_available_from_metric_type = country_info.get(TOURISM_AVAILABLE_FROM_METRIC_TYPE.name, None)
    if tourism_availability is None or tourism_available_from_metric_type is None:
        return []

    current_date = datetime.datetime.now().date()

    tourism_availability_date = datetime.datetime.strptime(tourism_available_from_metric_type.value, '%Y-%m-%d').date()
    if tourism_availability_date <= current_date and tourism_availability.value is False:
        return ['country_is_closed_but_open_date_is_reached']

    return []


def run_all_metric_correctness_validators(country_info: CountryInfo) -> List[str]:
    result = []
    for validator in metric_correctness_validators:
        result.extend(validator(country_info))
    return list(set(result))
