import logging
from typing import Dict

from travel.library.python.solomon_push_client import SolomonPushClient

from travel.avia.country_restrictions.lib.types import Environment

logger = logging.getLogger(__name__)


SOLOMON_PROJECT = 'avia'
SOLOMON_CLUSTER_BASE_NAME = 'country_restrictions'
SOLOMON_SERVICE = 'country_restrictions'


class SolomonPusher:
    internal_pusher: SolomonPushClient
    token = None

    def __init__(self, environment: Environment, token: str, parser_name: str):
        self.token = token
        self.parser_name = parser_name
        if token is None:
            logger.warning('No solomon token provided. No data will be sent to solomon')
        self.internal_pusher = SolomonPushClient(
            project=SOLOMON_PROJECT,
            cluster=self.get_solomon_cluster(environment),
            service=SOLOMON_SERVICE,
            token=token,
            labels={'environment': environment.value},
        )

    @staticmethod
    def get_solomon_cluster(env: Environment) -> str:
        return f'{SOLOMON_CLUSTER_BASE_NAME}_{env.value}'

    def push(self, sensor: str, value: int, labels: Dict[str, str]):
        if self.token is None:
            return
        labels['parser_name'] = self.parser_name
        self.internal_pusher.send(sensor, value, **labels)
        self.internal_pusher.upload()

    def __push_stats(self, value: int, data_type: str, state_type: str):
        self.push('stats', value, {'stats_data_type': data_type, 'stats_state_type': state_type})

    def push_stats(
        self,
        updated_geo: int,
        updated_metrics: int,
        updated_cells: int,
        null_geo: int,
        null_metrics: int,
        null_cells: int,
        total_geo: int,
        total_metrics: int,
        total_cells: int,
    ):
        self.__push_stats(updated_geo, 'geo', 'updated')
        self.__push_stats(updated_metrics, 'metrics', 'updated')
        self.__push_stats(updated_cells, 'cells', 'updated')

        self.__push_stats(null_geo, 'geo', 'null')
        self.__push_stats(null_metrics, 'metrics', 'null')
        self.__push_stats(null_cells, 'cells', 'null')

        self.__push_stats(total_geo, 'geo', 'total')
        self.__push_stats(total_metrics, 'metrics', 'total')
        self.__push_stats(total_cells, 'cells', 'total')

    def push_errors(self, errors: Dict[str, int]):
        for err_name, err_count in errors.items():
            if err_count == 0:
                continue
            self.push(f'errors.{err_name}', err_count, {})
