from dataclasses import dataclass
from enum import Enum
from pydantic import BaseModel
from typing import Dict, List

from travel.avia.country_restrictions.lib.geo_format_manager import GeoFormatManager
from travel.avia.country_restrictions.lib.table_format.base_format import BaseFormat
from travel.avia.country_restrictions.lib.table_format.one_column_metrics_base_format import OneColumnMetricsBaseFormat
from travel.avia.country_restrictions.lib.types import CountryInfo
from travel.avia.country_restrictions.lib.types.metric import ExtendedMetric
from travel.avia.country_restrictions.lib.types.metric_type import ENTRANCE_FOR_RUSSIANS, EntranceForRussiansEnum, \
    FLIGHTS_AVAILABILITY_V2, FlightAvailabilityV2Enum, IS_SPUTNIK_APPROVED, \
    PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED, TOURISM_AVAILABILITY, TOURISM_AVAILABLE_FROM_METRIC_TYPE, \
    QUARANTINE_REQUIRED, VISA_REQUIRED, AIRPORT_AVAILABILITY_IN_COUNTRY
from travel.avia.country_restrictions.lib.utils.multi_cluster_yt_client import get_yt_column_description


def get_metric_value_or_none(data: CountryInfo, metric_type):
    metric = data.get(metric_type.name, None)
    if metric is None:
        return None
    return metric.value


# https://wiki.yandex-team.ru/avia/dev/services-table/country-restrictions#informacijanasvernutombannere
NO_TOURISM_INFO_TEXT_SUFFIX = 'нет информации о туризме'
NO_TOURISM_INFO_MORE_TEXT = []

CLOSED_COUNTRY_TITLE_TEXT_SUFFIX = 'для туристов закрыто'
CLOSED_COUNTRY_SIMPLE_MORE_TEXT = ['Посещение страны с целью туризма временно недоступно']

OPENED_COUNTRY_TITLE_TEXT_SUFFIX = 'для путешествий открыто'


@dataclass
class MinimizedTextsInfo:
    title_text: str
    desktop_texts_info: List[str]
    mobile_texts_info: List[str]


class BannerColorEnum(Enum):
    GREEN = 'green'
    GRAY = 'gray'
    RED = 'red'


class ExtendedMetricsAnswer(BaseModel):
    color: str
    title_text: str
    desktop_minimized: List[str]
    mobile_minimized: List[str]
    expanded_banner_metrics: List[ExtendedMetric]


POINTS_FORCE_COLORS = {
    'l225': BannerColorEnum.GREEN,
}


# logic for colors, texts, etc is here:
# https://wiki.yandex-team.ru/users/nikhovas/countryinfo/logic/#informacijanarazvernutombannere
class ExtendedMetricsFormat(BaseFormat):
    YT_METRICS_AS_LIST_COLUMN_NAME: str = 'value'
    YT_METRICS_AS_LIST_COLUMN_TYPE: str = 'string'

    def __init__(self, output_table_short_name: str, metrics_list):
        super().__init__(output_table_short_name)
        self.metrics_list = metrics_list

    def get_yt_schema(self, metric_types) -> List[Dict[str, str]]:
        return [
            *self.indexes_column_descriptions,
            get_yt_column_description(self.YT_METRICS_AS_LIST_COLUMN_NAME, self.YT_METRICS_AS_LIST_COLUMN_TYPE),
        ]

    def dict_data_to_yt_format(self, data, geo_format_manager: GeoFormatManager):
        rows = []
        for key, values in data.items():
            geo_id = geo_format_manager.get_geo_id_by_point_key(key)
            if geo_id is None:
                continue

            banner_color = POINTS_FORCE_COLORS.get(key, None)
            if banner_color is None:
                banner_color = self.get_banner_color(values)
            texts = self.get_minimized_texts(key, values, banner_color, geo_format_manager)

            result = ExtendedMetricsAnswer(
                color=banner_color.value,
                title_text=texts.title_text,
                desktop_minimized=texts.desktop_texts_info,
                mobile_minimized=texts.mobile_texts_info,
                expanded_banner_metrics=self.process_metrics(values),
            )

            geo_result = result.json()

            row_value = {
                OneColumnMetricsBaseFormat.YT_POINT_KEY_COLUMN_NAME: key,
                OneColumnMetricsBaseFormat.YT_GEO_ID_COLUMN_NAME: geo_id,
                OneColumnMetricsBaseFormat.YT_METRICS_AS_LIST_COLUMN_NAME: geo_result,
            }
            rows.append(row_value)
        return rows

    def prepare_data(self, data, geo_format_manager: GeoFormatManager):
        result = {}
        for geo, d in data.items():
            if geo is None:
                continue
            result[geo] = d
        return self.dict_data_to_yt_format(result, geo_format_manager)

    @staticmethod
    def get_metric_value_or_none(data: CountryInfo, metric_type):
        metric = data.get(metric_type.name, None)
        if metric is None:
            return None
        return metric.value

    def process_metrics(self, data):
        result = []

        tourism_availability = ExtendedMetricsFormat.get_metric_value_or_none(data, TOURISM_AVAILABILITY)
        if tourism_availability is None:
            return []

        for metric_type in self.metrics_list:
            extended = metric_type.generate_extended_metric(data.get(metric_type.name, None))
            if extended is not None:
                result.append(extended)

        return result

    # Update https://wiki.yandex-team.ru/avia/dev/services-table/country-restrictions/#logikavyboracvetabannera
    # if you want to update this method
    @staticmethod
    def get_banner_color(data: CountryInfo):
        if data is None:
            return BannerColorEnum.GRAY

        tourism_availability = ExtendedMetricsFormat.get_metric_value_or_none(data, TOURISM_AVAILABILITY)
        flights_availability_v2 = ExtendedMetricsFormat.get_metric_value_or_none(data, FLIGHTS_AVAILABILITY_V2)
        entrance_for_russians = ExtendedMetricsFormat.get_metric_value_or_none(data, ENTRANCE_FOR_RUSSIANS)
        quarantine_required = ExtendedMetricsFormat.get_metric_value_or_none(data, QUARANTINE_REQUIRED)
        airport_availability = ExtendedMetricsFormat.get_metric_value_or_none(data, AIRPORT_AVAILABILITY_IN_COUNTRY)

        if airport_availability is False:
            flights_availability_v2 = FlightAvailabilityV2Enum.DIRECT_FLIGHTS.value.value

        if tourism_availability is False or \
           entrance_for_russians == EntranceForRussiansEnum.NO_ENTRANCE.value.value or \
           flights_availability_v2 == FlightAvailabilityV2Enum.NO_FLIGHTS.value.value:
            return BannerColorEnum.RED

        if (tourism_availability is True and quarantine_required is True) or tourism_availability is None:
            return BannerColorEnum.GRAY

        if entrance_for_russians in (
            EntranceForRussiansEnum.FREE_ENTRANCE.value.value,
            EntranceForRussiansEnum.HAS_RESTRICTIONS.value.value,
        ) and flights_availability_v2 in (
            FlightAvailabilityV2Enum.DIRECT_FLIGHTS.value.value,
            FlightAvailabilityV2Enum.TRANSFER_FLIGHTS.value.value,
        ):
            return BannerColorEnum.GREEN

        return BannerColorEnum.GRAY

    @staticmethod
    def add_metric_text_to_short_texts(
        data: CountryInfo,
        metric_name: str,
        minimized_texts_info: MinimizedTextsInfo,
        set_mobile_only_if_value_is_true: bool = False,
    ):
        metric_data = data.get(metric_name, None)
        if metric_data is not None:
            result_text = metric_data.text.to_simple_string_if_no_url()
            if result_text is None:
                return

            minimized_texts_info.desktop_texts_info.append(result_text)
            if metric_data.value or not set_mobile_only_if_value_is_true:
                minimized_texts_info.mobile_texts_info.append(result_text)

    @staticmethod
    def try_set_country_name_as_prefix(point_key: str, text: str, geo_format_manager: GeoFormatManager):
        country_name = geo_format_manager.get_point_ru_name_by_point_key(point_key)
        if country_name is None:
            return text.capitalize()
        else:
            return f'{country_name}: {text}'

    # Update https://wiki.yandex-team.ru/avia/dev/services-table/country-restrictions/#informacijanasvernutombannere
    # if you want to update this method
    @staticmethod
    def get_minimized_texts(
        point_key: str,
        data: CountryInfo,
        banner_color: BannerColorEnum,
        geo_format_manager: GeoFormatManager,
    ) -> MinimizedTextsInfo:
        tourism_availability = ExtendedMetricsFormat.get_metric_value_or_none(data, TOURISM_AVAILABILITY)
        airport_availability = ExtendedMetricsFormat.get_metric_value_or_none(data, AIRPORT_AVAILABILITY_IN_COUNTRY)
        if airport_availability is None:
            airport_availability = True

        if tourism_availability is None:
            return MinimizedTextsInfo(
                title_text=ExtendedMetricsFormat.try_set_country_name_as_prefix(
                    point_key,
                    NO_TOURISM_INFO_TEXT_SUFFIX,
                    geo_format_manager,
                ),
                desktop_texts_info=[],
                mobile_texts_info=[],
            )
        elif tourism_availability is False:
            opened_from = data.get(TOURISM_AVAILABLE_FROM_METRIC_TYPE.name, None)
            if opened_from is not None:
                return MinimizedTextsInfo(
                    title_text=ExtendedMetricsFormat.try_set_country_name_as_prefix(
                        point_key,
                        CLOSED_COUNTRY_TITLE_TEXT_SUFFIX,
                        geo_format_manager,
                    ),
                    desktop_texts_info=[opened_from.text.to_simple_string_if_no_url()],
                    mobile_texts_info=[opened_from.text.to_simple_string_if_no_url()],
                )

        if banner_color == BannerColorEnum.RED:
            return MinimizedTextsInfo(
                title_text=ExtendedMetricsFormat.try_set_country_name_as_prefix(
                    point_key,
                    CLOSED_COUNTRY_TITLE_TEXT_SUFFIX,
                    geo_format_manager,
                ),
                desktop_texts_info=CLOSED_COUNTRY_SIMPLE_MORE_TEXT,
                mobile_texts_info=CLOSED_COUNTRY_SIMPLE_MORE_TEXT,
            )

        result = MinimizedTextsInfo(
            title_text=ExtendedMetricsFormat.try_set_country_name_as_prefix(
                point_key,
                OPENED_COUNTRY_TITLE_TEXT_SUFFIX,
                geo_format_manager,
            ),
            desktop_texts_info=[],
            mobile_texts_info=[],
        )

        if airport_availability is True:
            ExtendedMetricsFormat.add_metric_text_to_short_texts(data, FLIGHTS_AVAILABILITY_V2.name, result)
        ExtendedMetricsFormat.add_metric_text_to_short_texts(data, QUARANTINE_REQUIRED.name, result, True)
        ExtendedMetricsFormat.add_metric_text_to_short_texts(data, IS_SPUTNIK_APPROVED.name, result)
        ExtendedMetricsFormat.add_metric_text_to_short_texts(
            data, PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED.name,
            result,
        )
        ExtendedMetricsFormat.add_metric_text_to_short_texts(data, VISA_REQUIRED.name, result, True)

        if len(result.mobile_texts_info) == 0:
            result.mobile_texts_info = result.desktop_texts_info

        return result
