from datetime import datetime, timezone
from typing import Any, List, Optional

from pydantic import BaseModel, Field
from pydantic.datetime_parse import parse_datetime

from travel.avia.country_restrictions.lib.types.rich_string import RichString


class UtcDatetime(datetime):
    @classmethod
    def __get_validators__(cls):
        yield parse_datetime
        yield cls.ensure_tzinfo

    @classmethod
    def ensure_tzinfo(cls, v):
        if v.tzinfo is None:
            return v.replace(tzinfo=timezone.utc)
        return v.astimezone(timezone.utc)

    @staticmethod
    def to_str(dt: datetime) -> str:
        return dt.isoformat()


def get_newest_ts(ts1: Optional[UtcDatetime], ts2: Optional[UtcDatetime]):
    if ts1 is None:
        return ts2
    elif ts2 is None:
        return ts1
    else:
        return max(ts1, ts2)


def union_meta_data_elem(elem1: str, elem2: str):
    if elem1 is None and elem2 is None:
        return None
    else:
        return f'[{elem1}, {elem2}]'


class Metric(BaseModel):
    value: Any
    text: RichString
    exclusions: List[RichString] = Field(default_factory=list)
    additions: List[RichString] = Field(default_factory=list)
    source: Optional[str] = None
    last_modification_time: Optional[UtcDatetime] = None
    updated_time: Optional[UtcDatetime] = None
    updated_user: Optional[str] = None
    point_key: Optional[str] = None

    class Config:
        json_encoders = {UtcDatetime: UtcDatetime.to_str}

    @staticmethod
    def equal_without_meta_info(l, r) -> bool:
        return l.value == r.value and l.text == r.text and l.exclusions == r.exclusions and l.additions == r.additions

    def union_metadata_from_another_metric(self, metric):
        self.source = union_meta_data_elem(self.source, metric.source)
        self.last_modification_time = get_newest_ts(self.last_modification_time, metric.last_modification_time)
        self.updated_time = get_newest_ts(self.updated_time, metric.updated_time)
        self.updated_user = union_meta_data_elem(self.updated_user, metric.updated_user)


class ExtendedMetric(BaseModel):
    title: RichString
    icon24: str
    metric_name: str
    metric_value: Optional[Metric]
