from pydantic import BaseModel
from typing import Optional

from travel.avia.country_restrictions.lib.types.rich_string import new_rich_text, new_rich_url
from travel.avia.country_restrictions.lib.types import Metric
from travel.avia.country_restrictions.lib.types.metric_type.metric_type import MetricType


class BankCardPaymentsMetricType(MetricType):
    class InternalData(BaseModel):
        russian_visa_mastercard: Optional[bool] = None
        mir: Optional[bool] = None
        union_pay: Optional[bool] = None

        def to_dict(self):
            return {
                'russianVisaMastercard': self.russian_visa_mastercard,
                'mir': self.mir,
                'unionPay': self.union_pay,
            }

    singular_work_prefix: str
    plural_work_prefix: str
    russian_visa_mastercard_text: str
    mir_text: str
    union_pay_text: str

    article_info_text: str
    article_info_link: str

    def generate_metric(
        self,
        value: Optional[InternalData],
        additions=None,
        exclusions=None,
        updated_user=None,
    ) -> Optional[Metric]:
        if additions is None:
            additions = []
        if exclusions is None:
            exclusions = []

        if value is None:
            return None

        if value.russian_visa_mastercard is not True and value.mir is not True and value.union_pay is not True:
            return None

        use_plural = False

        works_texts = []
        if value.russian_visa_mastercard is True:
            use_plural = True
            works_texts.append(self.russian_visa_mastercard_text)
        if value.mir is True:
            works_texts.append(self.mir_text)
        if value.union_pay is True:
            works_texts.append(self.union_pay_text)

        if len(works_texts) > 1:
            use_plural = True

        if use_plural:
            text = self.plural_work_prefix
        else:
            text = self.singular_work_prefix

        text += ' ' + ', '.join(works_texts)

        return Metric(
            value=value,
            text=new_rich_text(text),
            additions=additions,
            exclusions=exclusions,
            updated_user=updated_user,
        )

    @property
    def value_type(self) -> type:
        return dict

    def set_article_info(self, metric: Metric):
        metric.additions.append(new_rich_url(self.article_info_text, self.article_info_link))
