from typing import List, Optional

from travel.avia.country_restrictions.lib.types.rich_string import new_rich_text, RichString, TextBlock
from travel.avia.country_restrictions.lib.types import Metric
from travel.avia.country_restrictions.lib.types.metric_type.metric_type import MetricType


class ListMetricType(MetricType):
    singular_prefix_text: str
    plural_prefix_text: str
    trim_empty_elements: bool
    none_if_empty: bool
    empty_list_text: str

    def element_to_rich_string(self, elem: str, elem_number: int, total_number: int) -> RichString:
        if elem_number != total_number - 1:
            return new_rich_text(f'{elem}, ')
        else:
            return new_rich_text(elem)

    def generate_metric(
        self,
        value: Optional[List[str]],
        additions=None,
        exclusions=None,
        updated_user=None,
    ) -> Optional[Metric]:
        if additions is None:
            additions = []
        if exclusions is None:
            exclusions = []

        if not isinstance(value, list):
            return None

        if self.trim_empty_elements:
            value = [v for v in value if v is not None and v.strip('\n\t ') != '']

        if len(value) == 0:
            if self.none_if_empty:
                return None
            text = new_rich_text(self.empty_list_text)
        elif len(value) == 1:
            text = RichString(data=[
                TextBlock.create(self.singular_prefix_text + ' '),
                *self.element_to_rich_string(value[0], 0, 1).data,
            ])
        else:
            text = new_rich_text(self.plural_prefix_text + ' ')
            element_count = len(value)
            for i in range(element_count):
                text.extend(self.element_to_rich_string(value[i], i, element_count))

        text.compress()
        return Metric(value=value, text=text, additions=additions, exclusions=exclusions, updated_user=updated_user)

    @property
    def value_type(self) -> type:
        return List[str]
