from abc import abstractmethod
from enum import Enum
from pydantic import BaseModel, Field
from typing import Any, Optional

from travel.avia.country_restrictions.lib.types import Metric
from travel.avia.country_restrictions.lib.types.metric import ExtendedMetric
from travel.avia.country_restrictions.lib.types.rich_string import new_rich_text, RichString
from travel.avia.country_restrictions.lib.utils.multi_cluster_yt_client import get_yt_column_description


class FlagMetricExtensionType(Enum):
    ADDITION = 'addition'
    EXCLUSION = 'exclusion'


class FlagMetricExtension(BaseModel):
    text: RichString
    extension_type: FlagMetricExtensionType


class MetricType(BaseModel):
    name: str
    title: RichString
    icon24: str
    short_if_no_advanced_info: bool
    available_flag_extensions: Any = Field(default_factory=list)
    none_text_for_extended_metric: Optional[str] = None

    @property
    def yt_format(self):
        return get_yt_column_description(self.name, 'any')

    @property
    def value_type(self):
        return Any

    @abstractmethod
    def generate_metric(self, value: Any, additions=None, exclusions=None, changing_user=None) -> Optional[Metric]:
        pass

    def set_custom_addition(self, metric: Metric, text: RichString):
        metric.additions.append(text)

    def set_custom_exclusion(self, metric: Metric, text: RichString):
        metric.exclusions.append(text)

    def set_flag_extension(self, metric: Metric, flag: FlagMetricExtension):
        for value in self.available_flag_extensions:
            if value is flag:
                if flag.value.extension_type is FlagMetricExtensionType.ADDITION:
                    self.set_custom_addition(metric, flag.value.text)
                else:
                    self.set_custom_exclusion(metric, flag.value.text)
                break

    def generate_extended_metric(self, metric: Metric) -> Optional[ExtendedMetric]:
        if metric is None:
            return self.try_generate_extended_none_metric()

        if len(metric.exclusions) == 0 and len(metric.additions) == 0 and self.short_if_no_advanced_info is True:
            return ExtendedMetric(
                title=metric.text,
                icon24=self.icon24,
                metric_name=self.name,
                metric_value=None,
            )
        else:
            return ExtendedMetric(
                title=self.title,
                icon24=self.icon24,
                metric_name=self.name,
                metric_value=metric,
            )

    def try_generate_extended_none_metric(self) -> Optional[ExtendedMetric]:
        if self.none_text_for_extended_metric is not None:
            return ExtendedMetric(
                title=new_rich_text(self.none_text_for_extended_metric),
                icon24=self.icon24,
                metric_name=self.name,
                metric_value=None,
            )
        else:
            return None
