from typing import List, Literal, Optional, Union
from typing_extensions import Annotated

import copy
from pydantic import BaseModel, Field


TEXT_BLOCK_TYPE = 'text'


class TextBlock(BaseModel):
    class Element(BaseModel):
        text: str

    block_type: Literal['text'] = TEXT_BLOCK_TYPE
    data: Element

    @staticmethod
    def create(text: str):
        return TextBlock(data=TextBlock.Element(text=text))

    def get_string_description(self) -> str:
        return self.data.text

    def capitalize(self):
        self.data.text = self.data.text.capitalize()

    def lower(self):
        self.data.text = self.data.text.lower()


URL_BLOCK_TYPE = 'url'


class UrlBlock(BaseModel):
    class Element(BaseModel):
        text: str
        url: str

    block_type: Literal['url'] = URL_BLOCK_TYPE
    data: Element

    @staticmethod
    def create(text: str, url: str):
        return UrlBlock(data=UrlBlock.Element(text=text, url=url))

    def get_string_description(self) -> str:
        return f'[[{self.data.text}][{self.data.url}]]'

    def capitalize(self):
        self.data.text = self.data.text.capitalize()

    def lower(self):
        self.data.text = self.data.text.lower()


BlockUnionType = Union[TextBlock, UrlBlock]
RichStringBlock = Annotated[BlockUnionType, Field(discriminator='block_type')]


class RichString(BaseModel):
    data: List[RichStringBlock]

    @staticmethod
    def create(data: List[RichStringBlock]):
        rs = RichString(data=data)
        return rs.compressed()

    def compress(self):
        result = []
        blocks = self.data

        current_text = ''
        for block in blocks:
            if block.block_type == TEXT_BLOCK_TYPE:
                current_text += block.data.text
            elif current_text != '':
                result.append(TextBlock.create(current_text))
                result.append(block)
                current_text = ''

        if current_text != '':
            result.append(TextBlock.create(current_text))

        self.data = result

    def compressed(self):
        new_elem = RichString(data=copy.deepcopy(self.data))
        new_elem.compress()
        return new_elem

    def append(self, elem):
        if isinstance(elem, str):
            self.data.append(TextBlock.create(elem))
        else:
            self.data.append(elem)
        self.compress()

    def extend(self, data):
        if isinstance(data, RichString):
            self.data.extend(data.data)
        elif isinstance(data, list):
            self.data.extend(data)
        self.compress()

    def get_string_description(self) -> str:
        result_array = []
        for elem in self.data:
            result_array.append(elem.get_string_description())

        return ' '.join(result_array)

    def capitalize(self):
        if len(self.data) > 0:
            self.data[0].capitalize()
            for elem in self.data[1:]:
                elem.lower()

    def lower(self):
        for elem in self.data:
            elem.lower()

    def to_simple_string_if_no_url(self) -> Optional[str]:
        parts = []
        for elem in self.data:
            if elem.block_type == TEXT_BLOCK_TYPE:
                parts.append(elem.data.text)
            else:
                return None

        return ' '.join(parts)


def new_rich_text(text: str) -> RichString:
    return RichString(data=[TextBlock.create(text)])


def new_rich_url(text: str, url: str) -> RichString:
    return RichString(data=[UrlBlock.create(text=text, url=url)])
