import logging
from typing import Dict, List

import yt.wrapper as yt


class MultiClusterYtClient:
    def __init__(self, proxies: List[str], token=None):
        self.yt_clients: list = [yt.YtClient(proxy=proxy, token=token) for proxy in proxies]

    def one_success_operation_perform(self, op_name, *args, **kwargs):
        exc = None
        for client in self.yt_clients:
            try:
                return getattr(client, op_name)(*args, **kwargs)
            except Exception as err:
                exc = err

        raise exc

    def ignore_exceptions_operation_perform(self, op_name, *args, **kwargs):
        for client in self.yt_clients:
            try:
                return getattr(client, op_name)(*args, **kwargs)
            except Exception as exc:
                logging.getLogger().exception(exc)

    def read_table(self, table_path):
        return self.one_success_operation_perform('read_table', table_path)

    def write_table(self, table_path, rows):
        for client in self.yt_clients:
            client.write_table(table_path, rows)

    def list(self, folder, **kwargs):
        return self.one_success_operation_perform('list', folder, **kwargs)

    def get_attribute(self, table_name, attr_name):
        return self.one_success_operation_perform('get_attribute', table_name, attr_name)

    def remove(self, *args, **kwargs):
        return self.ignore_exceptions_operation_perform('remove', *args, **kwargs)

    def copy(self, *args, **kwargs):
        return self.ignore_exceptions_operation_perform('copy', *args, **kwargs)

    def create(self, *args, **kwargs):
        for client in self.yt_clients:
            client.create(*args, **kwargs)


def get_yt_column_description(column_name: str, column_type: str) -> Dict[str, str]:
    return {'name': column_name, 'type': column_type}
