from typing import Any, Dict, Optional, Union

from travel.avia.country_restrictions.lib.types import CountryInfo
from travel.avia.country_restrictions.lib.types.metric_type import (
    COVID_INSURANCE_REQUIRED,
    MOBILE_APP_REQUIRED,
    PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED,
    PcrExpirationPeriodInHoursOrRequiredExtensions,
)


def try_get_base_pcr(requirements: dict) -> Optional[bool]:
    return 'ПЦР' in requirements


def try_get_pcr_expiration_in_hours(row: Dict[str, Any]) -> Optional[int]:
    pcr_expiration = row.get('pcr_expiration', None)
    if pcr_expiration is None:
        return None

    pcr_time_text: str = pcr_expiration.get('ПЦР перед вылетом из РФ', None)
    if pcr_time_text is not None:
        try:
            hours = int(pcr_time_text.split(' ')[0])
            return hours
        except Exception:
            pass

    return None


def try_get_children_restrictions_from_requirements(children: dict) -> Union[int, bool, None]:
    """
    :return: inclusive age for kids when PCR is required, or True if it's required regardless
    """
    children_pcr_needed = children.get('pcr_needed', None)
    if children_pcr_needed == 'unknown':
        return None

    if children_pcr_needed is not None:
        age = children.get('starting_from_age', None)
        if age is not None:
            if age > 0:
                return age
            else:
                # if age is 0
                return True
        else:
            return False


# MAIN PARSER
def parser(context: CountryInfo, row: Dict[str, Any]) -> CountryInfo:
    requirements = row.get('requirements', None)
    children = row.get('children', None)
    if requirements is None:
        return {}

    covid_insurance_value = requirements.get('Обязательное страхование от коронавируса', None)
    mobile_app_value = requirements.get('Требуется загрузить приложение', None)

    result: CountryInfo = {
        COVID_INSURANCE_REQUIRED.name: COVID_INSURANCE_REQUIRED.generate_metric(covid_insurance_value),
        MOBILE_APP_REQUIRED.name: MOBILE_APP_REQUIRED.generate_metric(mobile_app_value),
    }

    no_vaccine_pcr_need = requirements.get('Не требуется вакцина/ПЦР или сведений о перенесенной болезни', False)
    if no_vaccine_pcr_need:
        result[PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED.name] = \
            PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED.generate_metric(False)
    elif isinstance(requirements, dict) and try_get_base_pcr(requirements):
        pcr_expiration_in_hours = try_get_pcr_expiration_in_hours(row)
        pcr_value = pcr_expiration_in_hours or True
        metric = PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED.generate_metric(pcr_value)

        children_pcr = try_get_children_restrictions_from_requirements(children)
        if children_pcr is False:
            PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED.set_flag_extension(
                metric,
                PcrExpirationPeriodInHoursOrRequiredExtensions.CHILDREN_FREE,
            )
        elif type(children_pcr) == int:
            PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED.set_children_pcr_free_before_age_text(metric, children_pcr)

        antigen = row.get('antigen_upon_arrival', None)
        if antigen is True:
            PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED.set_flag_extension(
                metric,
                PcrExpirationPeriodInHoursOrRequiredExtensions.ANTIGEN_TEST,
            )

        result[PCR_EXPIRATION_PERIOD_IN_HOURS_OR_REQUIRED.name] = metric

    return result
