from typing import Any, Dict, List

from travel.avia.country_restrictions.lib.types import CountryInfo
from travel.avia.country_restrictions.lib.types.metric_type import APPROVED_VACCINES, IS_SPUTNIK_APPROVED, \
    VACCINE_REQUIRED

SPUTNIK_TAGS = ['спутник', 'sputnik', 'гам-ковид-вак', 'гамковидвак']


def find_strings_in_string(to_find: List[str], string: str):
    lowercase = string.lower()
    for find_elem in to_find:
        if find_elem.lower() in lowercase:
            return True

    return False


def parser(context: CountryInfo, row: Dict[str, Any]) -> CountryInfo:
    requirements = row.get('requirements', None)
    if requirements is not None:
        no_vaccine_pcr_need = requirements.get('Не требуется вакцина/ПЦР или сведений о перенесенной болезни', False)
        if no_vaccine_pcr_need:
            return {
                VACCINE_REQUIRED.name: VACCINE_REQUIRED.generate_metric(False),
            }

    vaccines = row.get('vaccines', None)
    if vaccines is None:
        return {}

    req = vaccines.get('requirements', None)
    qua = vaccines.get('quarantine', None)
    allowed_vaccines_list = []
    quarantine_required_vaccines = []
    sputnik_approved = False
    sputnik_quarantine = None

    if req is not None:
        for key, value in req.items():
            if key != 'Другое':
                if value is True:
                    allowed_vaccines_list.append(key)
                if find_strings_in_string(SPUTNIK_TAGS, key):
                    sputnik_approved = value

    if qua is not None:
        for key, value in qua.items():
            if key != 'Другое':
                if value is True:
                    quarantine_required_vaccines.append(key)
                if find_strings_in_string(SPUTNIK_TAGS, key):
                    sputnik_quarantine = value

    vaccines_metric = APPROVED_VACCINES.generate_metric(allowed_vaccines_list)
    if vaccines_metric is None:
        return {}

    quarantine_required_vaccines = [x for x in quarantine_required_vaccines if x in allowed_vaccines_list]
    if len(quarantine_required_vaccines) != 0:
        APPROVED_VACCINES.set_quarantine_vaccines_text(vaccines_metric, quarantine_required_vaccines)

    sputnik_metric = IS_SPUTNIK_APPROVED.generate_metric(sputnik_approved)
    if sputnik_approved and sputnik_quarantine:
        IS_SPUTNIK_APPROVED.set_required_quarantine(sputnik_metric)

    return {
        VACCINE_REQUIRED.name: VACCINE_REQUIRED.generate_metric(True),
        APPROVED_VACCINES.name: vaccines_metric,
        IS_SPUTNIK_APPROVED.name: sputnik_metric,
    }
