from datetime import datetime
from typing import List, Optional

from travel.avia.country_restrictions.lib.table_format.base_format import BaseFormat
from travel.avia.country_restrictions.lib.table_format.metrics_as_columns_format import MetricsAsColumnsFormat
from travel.avia.country_restrictions.lib.parsers.yt_tables_to_yt_table_parser import YtTablesToYtTableParser
from travel.avia.country_restrictions.lib.types import CountryInfo
from travel.avia.country_restrictions.lib.types.metric_type import (
    FLIGHTS_AVAILABILITY,
    QUARANTINE_REQUIRED,
    TOURISM_AVAILABILITY,
    VISA_REQUIRED,
)
from travel.avia.country_restrictions.lib.types.metric_type.metric_type import MetricType


class SerpParser(YtTablesToYtTableParser):
    INPUT_YT_TABLE: str = '//home/rearrange/prism/travel/PRISM-68/filters'
    SOURCE_ID = 'serp'
    METRIC_TYPES: List[MetricType] = [QUARANTINE_REQUIRED, FLIGHTS_AVAILABILITY, TOURISM_AVAILABILITY, VISA_REQUIRED]

    UPDATING_TABLE_NAME = 'source-serp-result'
    OUTPUT_TABLE_FORMATS: List[BaseFormat] = [
        MetricsAsColumnsFormat(output_table_short_name='source-serp-result')
    ]

    PARSER_NAME = SOURCE_ID

    def get_input_table_names(self, data) -> List[str]:
        return [self.INPUT_YT_TABLE]

    def get_point_key(self, row) -> Optional[str]:
        geo_id = row['geo_id']
        return self.geo_format_manager.get_point_key_by_geo_id(geo_id)

    def parse_line(self, row, table_modification_time: datetime) -> CountryInfo:
        quarantine_info = row['quarantine_days'] or row['quarantine']
        data = {
            QUARANTINE_REQUIRED.name: QUARANTINE_REQUIRED.generate_metric(quarantine_info),
            FLIGHTS_AVAILABILITY.name: FLIGHTS_AVAILABILITY.generate_metric(row['avia']),
            TOURISM_AVAILABILITY.name: TOURISM_AVAILABILITY.generate_metric(row['tourism']),
            VISA_REQUIRED.name: VISA_REQUIRED.generate_metric(row['visa']),
        }

        for k, v in data.items():
            if v is not None:
                v.last_modification_time = table_modification_time

        return data
