#!/usr/bin/env python3
# encoding: utf-8

import os
import logging

import requests

PROJECT = 'avia'
ALERTS_DIR = 'alerts'
ENVS = ['production', 'testing']


def read_token(path):
    return


class SolomonClient(object):
    def __init__(self, project_id, auth_token_path=None, auth_token_env=None, auth_token=None):
        token = None
        if token is None and auth_token is not None:
            token = auth_token
        if token is None and auth_token_env is not None:
            token = os.environ.get(auth_token_env)
        if token is None and auth_token_path is not None:
            if os.path.exists(auth_token_path):
                with open(auth_token_path) as handle:
                    token = handle.read().strip()
        if token is None:
            raise ValueError("Missing Solomon token")
        self.session = requests.session()
        self.session.headers["Authorization"] = "OAuth %s" % token
        self.url_base = "https://solomon.yandex-team.ru/api/v2/projects/%s/" % project_id

    def url_for(self, *args):
        return self.url_base + "/".join(map(str, args))

    def get(self, *args):
        response = self.session.get(self.url_for(*args))
        if response.status_code == 200:
            return response.json()
        if response.status_code == 404:
            return None
        self.print_and_raise_for_status(response)

    def post(self, data, *args):
        response = self.session.put(self.url_for(*args), json=data)
        self.print_and_raise_for_status(response)

    def put(self, data, *args):
        response = self.session.put(self.url_for(*args), json=data)
        self.print_and_raise_for_status(response)

    def upsert(self, resource_name, resource_id, data):
        logging.debug("Upserting resource '%s' with id '%s'", resource_name, resource_id)
        old_data = self.get(resource_name, resource_id)
        if old_data is None:
            logging.debug("Resource '%s' with id '%s' not found; creating", resource_name, resource_id)
            self.post(data, resource_name, resource_id)
            logging.debug("Created resource '%s' with id '%s'", resource_name, resource_id)
        else:
            logging.debug("Resource '%s' with id '%s' already exists; updating", resource_name, resource_id)
            version = old_data.get("version")
            self.drop_system_fields(old_data)
            if old_data == data:
                logging.debug("Resource '%s' with id '%s' has not changed; skipping", resource_name, resource_id)
                return
            old_data.update(data)
            old_data["version"] = version
            self.put(old_data, resource_name, resource_id)
            logging.debug("Updated resource '%s' with id '%s'", resource_name, resource_id)

    @staticmethod
    def drop_system_fields(data):
        data.pop("updatedAt")
        data.pop("updatedBy")
        data.pop("createdAt")
        data.pop("createdBy")
        data.pop("version")
        data.pop("projectId")

    @staticmethod
    def print_and_raise_for_status(response):
        try:
            response.raise_for_status()
        except:
            print(response.content)
            raise


class SolomonAlerts(object):
    def __init__(self, auth_token_path):
        with open(auth_token_path) as f:
            auth_token = f.readlines()[0].strip()
        self.session = requests.session()
        self.session.headers['Authorization'] = 'OAuth %s' % auth_token
        self.url_base = "https://solomon.yandex-team.ru/api/v2/projects/%s/alerts" % PROJECT

    def alert_url(self, alert_id):
        return os.path.join(self.url_base, alert_id)

    def list_alerts(self):
        token = ""
        while True:
            response = self.session.get(self.url_base, params={"pageToken": token})
            self.print_and_raise_for_status(response)
            json = response.json()
            for item in json["items"]:
                yield item
            token = json.get("nextPageToken", "")
            if not token:
                break

    def get_alert(self, alert_id):
        response = self.session.get(self.alert_url(alert_id))
        if response.status_code == 200:
            return response.json()
        if response.status_code == 404:
            return None
        self.print_and_raise_for_status(response)

    def create_alert(self, alert):
        response = self.session.post(self.url_base, json=alert)
        self.print_and_raise_for_status(response)

    def update_alert(self, alert_id, alert):
        url = self.alert_url(alert_id)
        response = self.session.put(url, json=alert)
        self.print_and_raise_for_status(response)

    def write_single_alert(self, alert):
        alert_id = alert['id']
        alert_old = self.get_alert(alert_id)
        if alert_old is None:
            print('Alert %s not found. Creating new alert' % alert_id)
            self.create_alert(alert)
            print('Alert %s successfully created' % alert_id)
        else:
            version = alert_old['version']
            self.drop_system_fields(alert_old)
            if alert == alert_old:
                print("Alert %s not changed" % alert_id)
                return
            print('*** Alert %s changed, updating' % alert_id)
            alert['version'] = version
            self.update_alert(alert_id, alert)
            print('Alert %s successfully updated' % alert_id)

    def drop_system_fields(self, alert):
        SolomonClient.drop_system_fields(alert)
        alert.pop('state', None)

    @staticmethod
    def print_and_raise_for_status(response):
        SolomonClient.print_and_raise_for_status(response)
