#!/usr/bin/env python3
# encoding: utf-8

import argparse
import json
import logging
import os

from common import SolomonClient


def pull_objects_main(client, args):
    ids = args.ids
    object_type = args.object_type
    if not ids:
        ids = map(lambda item: item["id"], client.get(object_type)["result"])

    for id in ids:
        logging.debug("Pulling object of type %s: '%s'", object_type, id)
        datum = client.get(object_type, id)
        client.drop_system_fields(datum)

        os.makedirs(object_type, exist_ok=True)

        with open(os.path.join(object_type, id + ".json"), "w") as handle:
            json.dump(datum, handle, ensure_ascii=False, indent=2, sort_keys=True)


def push_objects_main(client, args):
    ids = args.ids
    object_type = args.object_type
    if not ids:
        raise ValueError("Missing object ids")

    for id in ids:
        object_path_prefix = "{}/".format(object_type)
        if id.startswith(object_path_prefix):
            id = id[len(object_path_prefix):]
        if id.endswith(".json"):
            id = id[:-len(".json")]
        logging.debug("Pushing object to %s id '%s'", object_type, id)
        with open(os.path.join(object_type, id + ".json"), "r") as handle:
            datum = json.load(handle)
            client.upsert(object_type, id, datum)


def main():
    logging.basicConfig(format="%(asctime)s - %(levelname)s - %(message)s", level=logging.DEBUG)

    parser = argparse.ArgumentParser()
    parser.add_argument("--project-id", default="travel")
    parser.add_argument("--auth-token-path", default=os.path.expanduser("~/.solomon/token"))
    parser.add_argument("--auth-token-env", default="SOLOMON_TOKEN")
    parser.add_argument("--auth-token")
    subparsers = parser.add_subparsers()

    manageable_object_types = ["services", "clusters"]

    pull_objects_parser = subparsers.add_parser("pull")
    pull_objects_parser.add_argument("object_type", choices=manageable_object_types, help="Object type (clusters|services)")
    pull_objects_parser.add_argument("--id", dest="ids", action="append")
    pull_objects_parser.set_defaults(func=pull_objects_main, ids=[])

    push_objects_parser = subparsers.add_parser("push")
    push_objects_parser.add_argument("object_type", choices=manageable_object_types, help="Object type (clusters|services)")
    push_objects_parser.add_argument("--id", dest="ids", action="append")
    push_objects_parser.set_defaults(func=push_objects_main, ids=[])

    args = parser.parse_args()
    client = SolomonClient(project_id=args.project_id,
                           auth_token_path=args.auth_token_path,
                           auth_token_env=args.auth_token_env,
                           auth_token=args.auth_token)
    args.func(client, args)


if __name__ == "__main__":
    main()
