#!/usr/bin/env python3
# encoding: utf-8

import argparse
import json
import os.path
import itertools
from collections import OrderedDict

import copy
from jinja2 import Template

import common

'''
Имя файла триггера должно быть в форме {serviceName}-{triggerName}.json

Триггер формируется при помощи шаблонов. Порядок применения:
1. Самый общий шаблон                        _base.json
2. Общий шаблон для dev, test и prod :       _base.{env}.json
3. Шаблон конкретного триггера               {trigger}.json
4. Перегрузки триггера для dev, test и prod  {trigger}.{env}.json

Для джагглера ключом алерта является пара host + service, поэтому все алерты обязаны иметь разные host+service.
Иначе они будут перетираться.

Соглашения для джагглера:
   tag = travel-avia - используется для аггрегации сырых событий
   host = cluster (для "общекластерных" алертов) или host для похостовых (low-space, например)
   service = имя_триггера + _service_add

   Для per-host мультиалертов нужно определить поле _host, равное "{{labels.host}}"
   Для мультиалертов с другим параметром группировки нужно определить поле _service_add с подстановкой, например
      "_service_add": "-{{labels.partner}}",
'''


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--auth-token-path', default=os.path.expanduser('~/.solomon/token'))
    parser.add_argument('--alert', action='append', default=[])
    parser.add_argument('--template', action='append', default=[])
    parser.add_argument('--env', action='append', default=[])
    args = parser.parse_args()
    if not args.env:
        args.env = common.ENVS

    if not args.alert:
        for f in os.listdir(common.ALERTS_DIR):
            if f.endswith('.json') and not f.startswith('_base.'):
                name = f[:-5]
                if not is_env_specific(name):
                    args.alert.append(name)
    if not args.template:
        for f in os.listdir(common.ALERTS_DIR):
            if f.endswith('.jinja2'):
                name = f[:-7]
                if not is_env_specific(name):
                    args.template.append(name)

    solomon = common.SolomonAlerts(args.auth_token_path)

    templated_alerts = {}
    for template_name in args.template:
        for env in args.env:
            merge_dicts(templated_alerts, build_alerts(template_name, env))

    for alert_name in args.alert:
        for env in args.env:
            alert = read_alert(alert_name, env, templated_alerts)
            only_env = alert.pop("only_env", None)
            if only_env is not None and only_env != env:
                continue
            fill_basic_data(alert, alert_name, env)
            solomon.write_single_alert(alert)
    for alert_name, alert_item in templated_alerts.items():
        for env in args.env:
            if alert_name.endswith(env):
                alert = copy.deepcopy(alert_item)
                only_env = alert.pop("only_env", None)
                if only_env is not None and only_env != env:
                    continue
                alert_name = alert_name[:-1 * (len(env) + 1)]
                fill_basic_data(alert, alert_name, env)
                solomon.write_single_alert(alert)


def merge_dicts(d1, d2):
    for k, v2 in d2.items():
        v1 = d1.get(k)
        if v1 is not None and isinstance(v1, dict):
            merge_dicts(v1, v2)
        else:
            d1[k] = v2


def read_alert(alert_name, env, templated_alerts):
    alert = templated_alerts.pop(alert_name, {})
    merge_dicts(alert, read_alert_for_env('_base', ''))
    merge_dicts(alert, read_alert_for_env('_base', env, {}))
    merge_dicts(alert, read_alert_for_env(alert_name, ''))
    merge_dicts(alert, read_alert_for_env(alert_name, env, {}))
    return alert


def build_alerts(template_name, env):
    alerts = {}
    merge_dicts(alerts, build_alerts_for_env(template_name, env, False, ''))
    merge_dicts(alerts, build_alerts_for_env(template_name, env, True, {}))
    res = {}
    for alert_name, alert_value in alerts.items():
        merge_dicts(alert_value, read_alert_for_env('_base', ''))
        merge_dicts(alert_value, read_alert_for_env('_base', env, {}))
        res[alert_name + '_' + env] = alert_value
    return res


def read_alert_for_env(alert_name, env, default=None):
    fn = alert_name
    if env:
        fn += '.' + env
    fn += '.json'
    path = os.path.join(common.ALERTS_DIR, fn)
    if os.path.exists(path):
        with open(path, 'rt') as f:
            return json.load(f)
    else:
        if default is not None:
            return default
        raise Exception('Alert not found by path %s' % path)


def build_alerts_for_env(template_name, env, env_specific, default=None):
    fn = template_name
    if env_specific:
        fn += '.' + env
    fn += ".jinja2"
    path = os.path.join(common.ALERTS_DIR, fn)
    if os.path.exists(path):
        with open(path, 'rt') as f:
            return template_alert(json.load(f), env)

    else:
        if default is not None:
            return default
        raise Exception('Template not found by path %s' % path)


def template_alert(template, env):
    res = {}
    variables = OrderedDict(template['variables'])
    keys = list(variables.keys())
    tpl = template['template']
    name = template['name']
    product = itertools.product(*variables.values())
    for p in product:
        context = dict(ENV=env)
        for i, v in enumerate(p):
            var = keys[i]
            context[var] = v
        res[render(name, context)] = render(tpl, context)
    return res


def render(obj, context):
    if isinstance(obj, str):
        return Template(obj).render(**context)
    elif isinstance(obj, list):
        return [render(o, context) for o in obj]
    elif isinstance(obj, dict):
        return {k: render(o, context) for k, o in obj.items()}
    else:
        return obj


def fill_basic_data(alert, alert_name, env):
    full_name = alert_name + '_' + env
    alert['id'] = full_name
    alert['name'] = full_name
    if alert['type'].get('expression') is not None:
        p = alert['type']['expression']['program']
        if isinstance(p, list):
            p = '\n'.join(p)
        to_replace = {'ENV': env}
        for k, v in to_replace.items():
            p = p.replace('{{%s}}' % k, v)
        alert['type']['expression']['program'] = p
    service_add = alert.pop('_service_add', '')
    host = alert.pop('_host', 'cluster')
    if 'groupByLabels' in alert and (service_add == '' and host == 'cluster'):
        raise Exception("When using groupByLabels (multialerts) you should also use _service_add or _host")
    alert['annotations']['env'] = env
    alert['annotations']['service'] = alert_name + service_add
    alert['annotations']['host'] = host


def is_env_specific(alert_name):
    for env in common.ENVS:
        if alert_name.endswith('.' + env):
            return True
    return False


if __name__ == '__main__':
    main()
