#!/usr/bin/env python

import os

JAVA_PATH = "/usr/bin/java"
BASE_LOG_DIR = "/ephemeral/var/log/yandex-avia-{{appName}}"
BASE_CACHE_DIR = "/ephemeral/var/cache"
SERVICE_NAME = "{{appName}}"
MAIN_CLASS = "{{mainClass}}"


def get_base_dir():
    return os.path.join(os.path.dirname(os.path.abspath(__file__)), "..")


def get_lib_dir():
    return os.path.join(get_base_dir(), SERVICE_NAME)


def get_classpath():
    jar_dir = os.path.join(SERVICE_NAME, "*")
    return jar_dir

def get_java_xmx_memory_limit():
    xmx_limit = os.environ.get('JAVA_MEMORY_LIMIT')
    if xmx_limit is None:
        xmx_limit = '2G'
    return xmx_limit



def get_java_args():
    return [
        "-DLOG_DIR={}".format(BASE_LOG_DIR),
        "-Djava.library.path={}".format(get_lib_dir()),

        # gc
        "-XX:MaxHeapSize={}".format(get_java_xmx_memory_limit()),
        "-verbose:gc",
        "-XX:+PrintGC",
        "-XX:+PrintGCDetails",
        "-Xloggc:{}/gc-%t.log".format(BASE_LOG_DIR),
        "-XX:ErrorFile={}/hs_err.log".format(BASE_LOG_DIR),
        "-XX:+HeapDumpOnOutOfMemoryError",
        "-XX:HeapDumpPath={}/hprof/".format(BASE_LOG_DIR),
        "-XX:OnOutOfMemoryError=kill",
        "-Xdebug",

        # ipv6
        "-Djava.net.preferIPv6Addresses=true",
        "-Djava.net.preferIPv4Stack=false",

        # i/o
        "-Dfile.encoding=UTF-8"
    ]


def get_profile():
    profile = os.environ.get("a_ctype")
    if profile is not None:
        return "-Dspring.profiles.active=" + profile


def get_yjp():
    if os.environ.get('ENABLE_PROFILER') == '1':
        return "-agentpath:/usr/lib64/libyjpagent.so=logdir=/dev/null"


def ensure_dirs():
    for dir in [BASE_LOG_DIR, BASE_CACHE_DIR]:
        if not os.path.exists(dir):
            os.makedirs(dir)


if __name__ == '__main__':
    ensure_dirs()
    args = get_java_args()
    args.append("-classpath")
    args.append(get_classpath())
    profile = get_profile()
    if profile:
        args.append(profile)
    yjp = get_yjp()
    if yjp:
        args.append(yjp)
    args.append(MAIN_CLASS)
    os.chdir(get_base_dir())
    os.execv(JAVA_PATH, args)

