# coding=utf-8

import logging
import os
import subprocess
import tempfile
import uuid

import yaml

logger = logging.getLogger(__name__)


class Releaser:
    DEFAULT_RELEASE_CAUSE = 'nightly'
    ENV_NAME = 'ZERO_DIFF_RELEASE_CAUSE'
    DEPLOY_UNIT_MARKER = object()
    DCTL_COMMAND = 'ya tool dctl'
    BOXES_PATH = ['spec', 'deploy_units', DEPLOY_UNIT_MARKER, 'replica_set', 'replica_set_template',
                  'pod_template_spec', 'spec', 'pod_agent_payload', 'spec']

    def __init__(self, stage, deploy_unit, box=None, workload=None, cause=None, dctl=None):
        self._stage = stage
        self._deploy_unit = deploy_unit
        self._box = box
        self._workload = workload

        if not (bool(self._box) ^ bool(self._workload)):
            raise ValueError('You should define one of "box" or "workload"')

        self._cause = cause or self.DEFAULT_RELEASE_CAUSE
        self._dctl_command = dctl or self.DCTL_COMMAND
        if isinstance(self._dctl_command, str):
            self._dctl_command = self._dctl_command.split()

    def run(self):
        logger.info('load spec for stage %r', self._stage)
        spec = yaml.full_load(self.run_dctl('get', 'stage', self._stage))
        payload = self.get_payload(spec)

        release_cause = 'sandbox_task_{}_{}'.format(self._cause, str(uuid.uuid4()))
        try:
            if self._box:
                target = self.get_box(payload)
            elif self._workload:
                target = self.get_workload(payload)
            else:
                raise RuntimeError
            self.set_env(target, release_cause)
            self.set_release_info(spec, release_cause)
        except Exception:
            logger.error('payload=%s', payload)
            raise

        with tempfile.NamedTemporaryFile(mode='w') as spec_file:
            logger.info('upload spec for stage %r', self._stage)
            yaml.dump(spec, spec_file)
            self.run_dctl('put', 'stage', spec_file.name)

    def get_payload(self, spec):
        current = spec
        current_path = []
        for key in self.BOXES_PATH:
            if key is self.DEPLOY_UNIT_MARKER:
                key = self._deploy_unit
            current_path.append(key)
            if key not in current:
                raise KeyError('Not found path ({!r}) in spec'.format('.'.join(current_path)))
            current = current[key]
        return current

    def get_box(self, boxes):
        return self.lookup_item_spec('box', boxes.get('boxes', []), self._box)

    def get_workload(self, box):
        return self.lookup_item_spec('workload', box.get('workloads', []), self._workload)

    @staticmethod
    def lookup_item_spec(item_type, items, key):
        for item in items:
            if item.get('id') == key:
                logger.info('found spec for %s %r', item_type, key)
                return item
        raise KeyError('Not found {} ({!r})'.format(item_type, key))

    def set_env(self, target_spec, release_cause):
        if 'env' not in target_spec:
            target_spec['env'] = []

        for env in target_spec['env']:
            if env['name'] == self.ENV_NAME:
                logger.info('previous release cause: %s', env['value']['literal_env']['value'])
                env['value']['literal_env']['value'] = release_cause
                return

        logger.info('new release cause: %s', release_cause)
        target_spec['env'].append({
            'name': self.ENV_NAME,
            'value': {
                'literal_env': {
                    'value': release_cause,
                },
            },
        })

    @staticmethod
    def set_release_info(spec, release_cause):
        spec['spec']['revision_info']['description'] = 'Auto release with cause={!r}'.format(release_cause)

    def run_dctl(self, *args):
        command = self._dctl_command + list(args)
        logger.info('dctl command: %s', ' '.join(command))

        p = subprocess.run(command, capture_output=True, env=os.environ.copy())
        if p.returncode != 0:
            raise RuntimeError(p.stderr)
        return p.stdout


if __name__ == '__main__':
    logger.setLevel(logging.INFO)
    logger.addHandler(logging.StreamHandler())

    releaser = Releaser('avia-wizard-unstable', 'app', workload='supervisor')
    releaser.run()
