import logging
import subprocess
import sys

import requests

AUTH_FILE = '/etc/nginx/auth/allow_yandex_only'
RACKTABLE_URL = 'https://ro.racktables.yandex-team.ru/export/expand-trypo-macros.php?macro=_YANDEXNETS_&skip_host64=1'
RACKTABLE_TIMEOUT = 5

try:
    import urllib3

    urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
    urllib3.disable_warnings(urllib3.exceptions.InsecurePlatformWarning)
    urllib3.disable_warnings(urllib3.exceptions.SNIMissingWarning)
except ImportError:
    pass


class RacktableError(Exception):
    pass


def get_subnets(racktable_url):
    s = requests.Session()
    a = requests.adapters.HTTPAdapter(max_retries=5)
    s.mount('', a)
    response = s.get(racktable_url, verify=False, timeout=RACKTABLE_TIMEOUT)
    if response.status_code != 200:
        raise RacktableError(response.text)

    for line in response.text.split('\n'):
        line = line.strip()
        if line:
            yield line

    s.close()


def reload_nginx():
    return subprocess.check_output(['supervisorctl', 'signal', 'HUP', 'nginx'])


def main():
    logging.info('Start')
    count = 0
    with open(AUTH_FILE, 'w') as auth_file:
        for subnet in get_subnets(RACKTABLE_URL):
            auth_file.write('allow {subnet};\n'.format(subnet=subnet))
            count += 1
        auth_file.write('allow 127.0.0.1;\n')
        auth_file.write('allow ::1;\n')
        auth_file.write('deny all;\n')

    logging.info('%d lines have been written', count)
    logging.info('reloading nginx')
    try:
        reload_output = reload_nginx()
        logging.info('%s', reload_output)
    except Exception:
        logging.exception('Failed to restart nginx')
    logging.info('End')



if __name__ == '__main__':
    logging.basicConfig(
        format='%(asctime)s %(levelname)s %(message)s',
        datefmt='%m/%d/%Y %H:%M:%S',
        stream=sys.stdout,
        level=logging.INFO,
    )
    main()
