import json
import os
from subprocess import Popen, PIPE


class PushClient(object):
    def __init__(self, logger, registry):
        self._logger = logger

    def pull(self, ts, consumer):
        status = self._get_status()
        self._push_result(status, ts, consumer)

    def _get_status(self):
        process = Popen([
            '/usr/bin/timeout', '-k', '10', '-s', '9', '10',
            '/usr/bin/push-client', '-f', '-c', self._get_push_client_config_file(), '--status', '--json',
        ], stdout=PIPE, stderr=PIPE)

        return_code = process.wait()
        status, stderr = process.stdout.read(), process.stderr.read()
        if return_code != 0:
            self._logger.error('push-client status failed, code: {}, stdout: {}, stderr: {}'.format(
                return_code,
                status,
                stderr,
            ))

            return []

        return json.loads(status)

    @staticmethod
    def _get_push_client_config_file():
        env = os.getenv('QLOUD_ENVIRONMENT')
        return '/etc/yandex/statbox-push-client/{env}/push-client.yaml'.format(env=env)

    def _push_result(self, status, ts, consumer):
        for log_description in status:
            log_name = os.path.basename(log_description['actual name']).split('.')[0].encode('utf8')

            for sensor in ('commit delay', 'lag'):
                value = log_description.get(sensor, None)
                if value is None:
                    self._logger.error('Not found value for {}'.format(sensor))
                else:
                    consumer.gauge(
                        {'sensor': sensor.replace(' ', '-'),  'log_name': log_name}, ts, value,
                    )

            last_send_time = log_description.get('last send time')
            if last_send_time is None:
                self._logger.error('Not found value for last send time')
            else:
                # Solomon gives timestamp in milliseconds
                time_delay = (ts / 1000) - log_description['last send time']
                consumer.gauge(
                    {'sensor': 'time-delay', 'log_name': log_name}, ts, time_delay,
                )
