package main

import (
	"context"
	"fmt"
	"net/http"
	"sync"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/avia/fare_families/internal/app/api"
	"a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/dicts"
	httpdump "a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/handlers/http/dump"
	httptariff "a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/handlers/http/tariff"
	"a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/storage"
	tariffmatcher "a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/tariff_matcher"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/library/go/configuration"
	dynamicdicts "a.yandex-team.ru/travel/library/go/dicts/updaterservice"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
)

const (
	serviceName  = "fare-families-api"
	ProfilerPort = 8181
)

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &api.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}
	cfg := api.Cfg

	logger, err := logging.NewDeploy(&cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	} else {
		logger.Info("loaded configuration", log.Reflect("config", cfg))
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		ctxCancel()
	}()

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, metricsserver.DefaultMetricsConfig.PerfMetricsRefreshInterval)

	// bootstrapping business logic
	storage, err := storage.NewStorageFromFiles(cfg.Storage, logger)
	if err != nil {
		logger.Fatal("Error while caching fare families conditions", log.Error(err))
	}
	if len(storage.GetFiles()) == 0 {
		logger.Fatal("No fare-family files are loaded, check you CONFIG_PATH")
	}

	dicts, err := dicts.NewRegistry(cfg.Dicts, logger)
	if err != nil {
		logger.Fatal("Error while loading dicts", log.Error(err))
		return
	}

	tariffMatcher, err := tariffmatcher.NewTariffMatcher(storage, dicts, logger)
	if err != nil {
		logger.Fatal("Error while creating a tariff matcher", log.Error(err))
		return
	}

	dumpHTTPHandler := httpdump.NewDumpHandler(storage, logger)
	tariffHTTPHandler := httptariff.NewTariffHandler(tariffMatcher, logger)
	tariffDebugHTTPHandler := httptariff.NewTariffDebugHandler(tariffMatcher, logger)
	filesListHandler := httpdump.NewFilesListHandler(storage, logger)

	httpRouteBuilders := []httpserver.RouteBuilder{
		dumpHTTPHandler.GetRouteBuilder(),
		tariffHTTPHandler.GetRouteBuilder(),
		tariffDebugHTTPHandler.GetRouteBuilder(),
		filesListHandler.GetRouteBuilder(),
	}

	// Metrics server
	go func() {
		err = metricsserver.RunMetricsHTTPServer(context.Background(), metricsserver.DefaultMetricsConfig, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	// add readiness and shutdown hooks
	probeState := probes.NewState(logger)
	probeRouteBuilders := probes.GetChiRouteBuilders(&cfg.Probes, probeState)
	httpRouteBuilders = append(httpRouteBuilders, probeRouteBuilders...)

	// running server
	wg := sync.WaitGroup{}
	wg.Add(1)

	// run dynamic dicts resources updater
	runDynamicDictsService(logger, dicts, cfg.DictsUpdater)

	go runProfiler(ProfilerPort, logger)
	go func() {
		defer wg.Done()
		server := httpserver.NewHTTPServerBuilder(cfg.HTTP, logger, httpRouteBuilders, rootRegistry.WithPrefix("http")).
			WithLoggingRecoverer().
			Build()
		err = server.Run(context.Background())
		if err != nil {
			logger.Fatal("Error while starting HTTP server", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()
}

func runProfiler(port int, logger log.Logger) {
	router := chi.NewRouter()
	router.Mount("/debug", middleware.Profiler())
	logger.Fatal("Server error",
		log.Error(
			http.ListenAndServe(
				fmt.Sprintf(":%d", port),
				router,
			),
		),
	)
}

func runDynamicDictsService(logger log.Logger, dictsRegistry *dicts.Registry, serviceConfig dynamicdicts.Config) {
	dynamicdicts.NewService(logger, serviceConfig, dictsRegistry.OnUpdate).BackroundRun()
}
