package farefamilies

import (
	"encoding/json"
	"regexp"

	"a.yandex-team.ru/travel/library/go/containers"
)

type FareFamily struct {
	BaseClass                string            `json:"base_class"`
	Brand                    string            `json:"brand"`
	TariffCodePattern        string            `json:"tariff_code_pattern"`
	NegatedTariffCodePattern string            `json:"negated_tariff_code_pattern,omitempty"`
	TariffGroupName          map[string]string `json:"tariff_group_name"`
	Terms                    []FareFamilyTerm  `json:"terms"`
	Key                      string            `json:"key,omitempty"`
}

type FareFamilyTerm struct {
	Code         string               `json:"code"`
	SpecialNotes []map[string]string  `json:"special_notes,omitempty"`
	Rules        []FareFamilyTermRule `json:"rules"`
}

type FareFamilyTermRule struct {
	Availability      string              `json:"availability"`
	Charge            *Charge             `json:"charge,omitempty"`
	Comment           string              `json:"comment,omitempty"`
	Places            int32               `json:"places,omitempty"`
	Size              string              `json:"size,omitempty"`
	SpecialNotes      []map[string]string `json:"special_notes,omitempty"`
	Weight            int32               `json:"weight,omitempty"`
	XPath             string              `json:"xpath,omitempty"`
	Conditions        []Condition         `json:"conditions,omitempty"`
	CarryOnSizeBucket string              `json:"carry_on_size_bucket,omitempty"`
	Miles             string              `json:"miles,omitempty"`
}

type Charge struct {
	Currency string `json:"currency,omitempty"`
	Value    string `json:"value,omitempty"`
}

// Parts of a condition that have default values are ignored
type Condition struct {
	AirportsOnly    []string `json:"airports,omitempty"`
	AirportsBetween Between  `json:"airports_between,omitempty"`
	CountriesFrom   []string `json:"countries_from,omitempty"`
	CountriesTo     []string `json:"countries_to,omitempty"`
	CountriesOnly   []string `json:"countries,omitempty"`
	IsDomesticRU    bool     `json:"is_domestic,omitempty"`
	IsInternational bool     `json:"is_international,omitempty"`
	DepartBefore    string   `json:"depart_before,omitempty"`
	DepartAfter     string   `json:"depart_after,omitempty"`
}

type Between struct {
	PointsA []string `json:"points_a,omitempty"`
	PointsB []string `json:"points_b,omitempty"`
}

type CompiledFareFamily struct {
	FareFamily
	TariffCodeRegExp *regexp.Regexp
	NegateRegexp     bool
}

type FareFamilyForVariant struct {
	BaseClass         string                     `json:"base_class"`
	Brand             string                     `json:"brand"`
	TariffCodePattern string                     `json:"tariff_code_pattern"`
	TariffGroupName   map[string]string          `json:"tariff_group_name"`
	Terms             []FareFamilyTermForVariant `json:"terms"`
	Key               string                     `json:"key,omitempty"`
}

type FareFamilyTermForVariant struct {
	ID           int                 `json:"id"`
	Code         string              `json:"code"`
	SpecialNotes []map[string]string `json:"special_notes,omitempty"`
	Rule         FareFamilyTermRule  `json:"rule"`
}

// Overriding to always output "places" field for the baggage rules
func (t *FareFamilyTermForVariant) MarshalJSON() ([]byte, error) {
	type FareFamilyTermForVariantAlias FareFamilyTermForVariant
	if t.Code != BaggageCode && t.Code != CarryOnCode {
		return json.Marshal((*FareFamilyTermForVariantAlias)(t))
	}

	type FareFamilyBaggageTermRule = struct {
		Places int32 `json:"places"`
		*FareFamilyTermRule
	}
	rule := FareFamilyBaggageTermRule{
		Places:             t.Rule.Places,
		FareFamilyTermRule: (*FareFamilyTermRule)(&t.Rule),
	}
	return json.Marshal(&struct {
		Rule *FareFamilyBaggageTermRule `json:"rule"`
		*FareFamilyTermForVariantAlias
	}{
		Rule:                          (*FareFamilyBaggageTermRule)(&rule),
		FareFamilyTermForVariantAlias: (*FareFamilyTermForVariantAlias)(t),
	})
}

var AllowedBaseClasses = containers.SetOf(
	"ECONOMY",
	"BUSINESS",
)

const (
	BaggageCode          = "baggage"
	CarryOnCode          = "carry_on"
	ChangingCarriageCode = "changing_carriage"
	RefundableCode       = "refundable"
)

var AllowedTermCodes = containers.SetOf(
	BaggageCode,
	CarryOnCode,
	ChangingCarriageCode,
	"changing_carriage_no_show",
	"disclosure_url",
	"miles",
	"open_return_date",
	RefundableCode,
	"refundable_no_show",
	"seat_selection_check_in",
)

type Availability string

const (
	NotAvailable       Availability = "NOT_AVAILABLE"
	AvailableForCharge Availability = "CHARGE"
	AvailableForFree   Availability = "FREE"
)

var AllowedAvailabilityValues = containers.SetOf(
	string(NotAvailable),
	string(AvailableForCharge),
	string(AvailableForFree),
)
