package dicts

import (
	"fmt"
	"io"
	"path/filepath"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/dicts/base"
)

type Config struct {
	ResourcesPath          string `config:"DICTS_RESOURCES_PATH" yaml:"resources_path"`
	UseDynamicResources    bool   `yaml:"use_dynamic_resources"`
	UpdateDynamicResources bool   `config:"UPDATE_DYNAMIC_RESOURCES" yaml:"update_dynamic_resources"`
}

var DefaultConfig = Config{
	ResourcesPath:          "/app/data",
	UseDynamicResources:    false,
	UpdateDynamicResources: true,
}

// TODO(u-jeen): когда сделаем RASPTICKETS-22376, перевести работу со справочниками на единую библиотеку
type Registry struct {
	writerByPath map[string]UpdatableWriter
	carriers     *CarriersRepository
	countries    *CountriesRepository
	stationCodes *StationCodesRepository
	stations     *StationsRepository
	config       Config
	logger       log.Logger
}

type DictsRegistry interface {
	Carriers() *CarriersRepository
	Countries() *CountriesRepository
	Stations() *StationsRepository
	StationCodes() *StationCodesRepository
}

func (r *Registry) Countries() *CountriesRepository {
	return r.countries
}

func (r *Registry) Stations() *StationsRepository {
	return r.stations
}

func (r *Registry) StationCodes() *StationCodesRepository {
	return r.stationCodes
}

func (r *Registry) Carriers() *CarriersRepository {
	return r.carriers
}

type RepositoryUpdater interface {
	Populate(writer io.Writer) error
}

type UpdatableWriter interface {
	io.Writer
	UpdateFromSource(RepositoryUpdater) error
}

func populateMany(path2writer map[string]UpdatableWriter) error {
	for path, writer := range path2writer {
		bytesIterator, err := base.BuildIteratorFromFile(path)
		if err != nil {
			return fmt.Errorf("internal.dicts.registry:populateMany %w", err)
		}
		err = bytesIterator.Populate(writer)
		if err != nil {
			return fmt.Errorf("internal.dicts.registry:populateMany %w", err)
		}
	}
	return nil
}

func NewRegistry(c Config, logger log.Logger) (*Registry, error) {
	r := &Registry{
		carriers:     NewCarriersRepository(),
		countries:    NewCountriesRepository(),
		stations:     NewStationsRepository(),
		stationCodes: NewStationCodesRepository(),
		config:       c,
		logger:       logger,
	}

	resourcePath, err := getResourcePath(c)
	if err != nil {
		return nil, fmt.Errorf("internal.dicts:NewRegistry %w", err)
	}

	r.updateWriterByPath(resourcePath)
	err = populateMany(r.writerByPath)
	if err != nil {
		return nil, fmt.Errorf("internal.dicts:NewRegistry %w", err)
	}

	if logger != nil {
		logger.Info(
			"Loaded dicts",
			log.Int("carriers_count", r.carriers.Size()),
			log.Int("station_codes_iata_count", r.stationCodes.SizeIATA()),
		)
	}
	return r, nil
}

func NewRegistryForTests() *Registry {
	r := &Registry{
		carriers:     NewCarriersRepository(),
		countries:    NewCountriesRepository(),
		stations:     NewStationsRepository(),
		stationCodes: NewStationCodesRepository(),
	}
	return r
}

func (r *Registry) OnUpdate() error {
	if !r.config.UseDynamicResources || !r.config.UpdateDynamicResources {
		return nil
	}

	resourcePath, err := getResourcePath(r.config)
	if err != nil {
		return fmt.Errorf("internal.dicts.registry:OnUpdate %w", err)
	}

	r.updateWriterByPath(resourcePath)

	for path, writer := range r.writerByPath {
		bytesIterator, err := base.BuildIteratorFromFile(path)
		if err != nil {
			return fmt.Errorf("internal.dicts.registry:OnUpdate %w", err)
		}
		if err := writer.UpdateFromSource(bytesIterator); err != nil {
			return fmt.Errorf("internal.dicts.registry:OnUpdate %w", err)
		}
	}
	if r.logger != nil {
		r.logger.Info(
			"Updated dicts",
			log.Int("carriers_count", r.carriers.Size()),
			log.Int("station_codes_iata_count", r.stationCodes.SizeIATA()),
		)
	}
	return nil
}

func (r *Registry) updateWriterByPath(resourcePath string) {
	r.writerByPath = map[string]UpdatableWriter{
		resourcePath + "/country.data":      r.countries,
		resourcePath + "/station_code.data": r.stationCodes,
		resourcePath + "/station.data":      r.stations,
		resourcePath + "/carrier.data":      r.carriers,
	}
}

func getResourcePath(config Config) (string, error) {
	resourcePath := config.ResourcesPath
	if config.UseDynamicResources {
		symlinkTarget, err := filepath.EvalSymlinks(config.ResourcesPath)
		if err != nil {
			return "", err
		}
		resourcePath = filepath.Dir(symlinkTarget)
	}
	return resourcePath, nil
}
