package tariff

import (
	"bytes"
	"compress/gzip"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/httputil/headers"
	"a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/data_structs/payloads"
	tariffmatcher "a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/tariff_matcher"
	"a.yandex-team.ru/travel/library/go/httputil"
)

type TariffHandler struct {
	tariffMatcher tariffmatcher.TariffMatcher
	logger        log.Logger
}

func NewTariffHandler(tariffMatcher tariffmatcher.TariffMatcher, logger log.Logger) *TariffHandler {
	return &TariffHandler{tariffMatcher: tariffMatcher, logger: logger}
}

func (h *TariffHandler) Handle(w http.ResponseWriter, r *http.Request) {
	span, _ := opentracing.StartSpanFromContext(
		r.Context(),
		"internal.services.fare_families.handler.http:TariffHandler",
	)
	defer span.Finish()
	start := time.Now()

	readBodySpan, _ := opentracing.StartSpanFromContext(
		r.Context(),
		"internal.services.fare_families.handler.http:TariffHandler:ReadBody",
	)
	var bodyReader io.ReadCloser
	if r.Header.Get(headers.ContentEncodingKey) == "gzip" {
		var ok bool
		var err error
		bodyReader, ok = r.Body.(*gzip.Reader)
		if !ok {
			bodyReader, err = gzip.NewReader(r.Body)
			if err != nil {
				httputil.HandleError(err, http.StatusBadRequest, w)
			}
		}
	} else {
		bodyReader = r.Body
	}
	body, err := ioutil.ReadAll(bodyReader)
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	err = r.Body.Close()
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	readBodySpan.Finish()

	createDecoderSpan, _ := opentracing.StartSpanFromContext(
		r.Context(),
		"internal.services.fare_families.handler.http:TariffHandler:CreateDecoder",
	)
	decoder := json.NewDecoder(bytes.NewReader(body))
	createDecoderSpan.Finish()

	decodeSpan, _ := opentracing.StartSpanFromContext(
		r.Context(),
		"internal.services.fare_families.handler.http:TariffHandler:Decode",
	)
	tariffHandlerPayload := &TariffHandlerPayload{}
	err = decoder.Decode(&tariffHandlerPayload)
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	decodeSpan.Finish()

	resultBytes, err := h.iterateOverVariants(tariffHandlerPayload, r.Context())
	defer func() {
		statsSpan, _ := opentracing.StartSpanFromContext(
			r.Context(),
			"internal.services.fare_families.handler.http:TariffHandler:Stats",
		)
		defer statsSpan.Finish()
		go func() {
			for _, variantsPack := range tariffHandlerPayload.VariantsFromPartner {
				h.logger.Info(
					fmt.Sprintf("Processed tariffs for variants pack in %s", time.Since(start)),
					log.String("qid", variantsPack.QueryID),
					log.Int32("all_variants_count", variantsPack.AllVariantsCount),
					log.Int("resultBytes", len(resultBytes)),
					log.Duration("elapsed", time.Since(start)),
					log.Error(err),
				)
			}
		}()
	}()

	writeSpan, _ := opentracing.StartSpanFromContext(
		r.Context(),
		"internal.services.fare_families.handler.http:TariffHandler:WriteResponse",
	)
	if err != nil {
		h.handleError(err, "unable to open write span", w)
		return
	}
	w.Header().Set("Content-Type", "application/json")
	if r.Header.Get(headers.AcceptEncodingKey) == "gzip" {
		w.Header().Set("Content-Encoding", "gzip")
	}
	w.WriteHeader(http.StatusOK)
	if r.Header.Get(headers.AcceptEncodingKey) == "gzip" {
		gw, err := gzip.NewWriterLevel(w, 5)
		if err != nil {
			h.handleError(err, "unable to create gzip writer", w)
			return
		}
		_, err = gw.Write(resultBytes)
		if err != nil {
			h.handleError(err, "unable to write gzipped results", w)
			return
		}
		err = gw.Close()
		if err != nil {
			h.handleError(err, "unable to close gzip writer", w)
			return
		}
	} else {
		_, _ = w.Write(resultBytes)
	}

	writeSpan.Finish()
}

func (h *TariffHandler) handleError(err error, errorMessage string, w http.ResponseWriter) {
	h.logger.Error(errorMessage, log.Error(err))
	httputil.HandleError(err, http.StatusInternalServerError, w)
}

func (h *TariffHandler) iterateOverVariants(tariffHandlerPayload *TariffHandlerPayload, context context.Context) ([]byte, error) {
	span, _ := opentracing.StartSpanFromContext(
		context,
		"internal.services.fare_families...:iterateOverVariants",
	)
	defer span.Finish()
	result := &TariffHandlerResponse{
		FareFamiliesMap: []payloads.FareFamiliesMap{},
	}
	for _, variantsPack := range tariffHandlerPayload.VariantsFromPartner {
		fareFamiliesMap, err := h.tariffMatcher.MatchFareFamilies(&variantsPack, context)
		if err != nil {
			return nil, err
		}
		result.FareFamiliesMap = append(result.FareFamiliesMap, *fareFamiliesMap)
	}
	resultBytes, err := json.Marshal(result)
	return resultBytes, err
}

func (h *TariffHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Post("/fare-families/tariff", h.Handle)
	}
}
