package tariff

import (
	"encoding/json"
	"net/http"

	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	tariffmatcher "a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/tariff_matcher"
	"a.yandex-team.ru/travel/library/go/httputil"
)

type TariffDebugHandler struct {
	tariffMatcher tariffmatcher.TariffMatcher
	logger        log.Logger
}

func NewTariffDebugHandler(tariffMatcher tariffmatcher.TariffMatcher, logger log.Logger) *TariffDebugHandler {
	return &TariffDebugHandler{tariffMatcher: tariffMatcher, logger: logger}
}

func (h *TariffDebugHandler) Handle(w http.ResponseWriter, r *http.Request) {
	span, _ := opentracing.StartSpanFromContext(
		r.Context(),
		"internal.services.fare_families.handler.http:TariffDebugHandler",
	)
	defer span.Finish()

	form := &TariffDebugHandlerForm{}
	if err := form.Bind(r); err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}

	fareFamily, err := h.tariffMatcher.MatchFareFamily(form.Carrier, form.FareCode, form.From, form.To, form.Departure)

	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	if fareFamily == nil {
		httputil.HandleError(xerrors.Errorf("No tariff match"), http.StatusNotFound, w)
		return
	}
	resultBytes, err := json.Marshal(TariffDebugHandlerResponse{*fareFamily})
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(http.StatusOK)
	_, _ = w.Write(resultBytes)
}

func (h *TariffDebugHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Get("/match", h.Handle)
	}
}
