package storage

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"regexp"
	"strconv"
	"strings"

	"a.yandex-team.ru/library/go/core/log"

	farefamiliesstructs "a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/data_structs/fare_families"
)

const (
	FareFamiliesFolder    = "fare_families"
	JSONFileNameSeparator = "_"
	JSONSuffix            = ".json"
)

type Storage interface {
	GetDump(carrier int32, filters ...FareFamilyFilterFunc) (dump *CarrierFareFamilies, err error)
	GetFiles() map[string]JSONFileStats
}

type StorageForTests interface {
	Storage
	AddFareFamily(carrier int32, fareFamily farefamiliesstructs.FareFamily) error
}

type CarrierFareFamilies struct {
	CarrierID    int32                                    `json:"carrier"`
	FareFamilies []farefamiliesstructs.CompiledFareFamily `json:"fare_families"`
}

type JSONFileStats struct {
	CarrierID         int32 `json:"carrier"`
	FareFamiliesCount int   `json:"fare_families_count"`
}

type storageImpl struct {
	data   map[int32]CarrierFareFamilies
	files  map[string]JSONFileStats
	logger log.Logger
}

// Returns true if the specified fare family matches this filter, false otherwise
type FareFamilyFilterFunc func(fareFamily farefamiliesstructs.CompiledFareFamily) bool

func NewStorageFromFiles(config StorageConfig, logger log.Logger) (Storage, error) {
	storageInstance := &storageImpl{
		data:   map[int32]CarrierFareFamilies{},
		files:  map[string]JSONFileStats{},
		logger: logger,
	}

	logger.Infof("Reading fares data: %s", config.Datapath)
	err := filepath.Walk(config.Datapath, func(path string, info os.FileInfo, err error) error {
		logger.Infof("Reading fares data file: %s", path)
		if err != nil {
			logger.Warnf("Error while reading fares data file: %r", err)
			return err
		}
		if info.IsDir() {
			return nil
		}
		if !strings.HasSuffix(path, JSONSuffix) {
			return nil
		}

		pathParts := strings.Split(strings.TrimSuffix(path, JSONSuffix), "/")
		filename := pathParts[len(pathParts)-1]
		filenameParts := strings.Split(filename, JSONFileNameSeparator)
		if len(filenameParts) >= 2 {
			bytes, err := ioutil.ReadFile(path)
			if err != nil {
				logger.Warnf("Error while reading fares data file %s: %r", path, err)
				return err
			}
			fareFamiliesData := []farefamiliesstructs.FareFamily{}
			if err := json.Unmarshal(bytes, &fareFamiliesData); err != nil {
				logger.Errorf("Error while reading fares data file %s: %r", path, err)
				return err
			}
			fareFamiliesCompiled := make([]farefamiliesstructs.CompiledFareFamily, len(fareFamiliesData))
			for idx, ffData := range fareFamiliesData {
				compiledffData, err := InitFareFamily(ffData)
				if err != nil {
					return err
				}
				fareFamiliesCompiled[idx] = *compiledffData
			}
			carrierCode, err := strconv.Atoi(filenameParts[0])
			if err != nil {
				logger.Warnf("Error while parsing file name %s: %r", path, err)
				return err
			}
			storageInstance.files[strings.Join(filenameParts[1:], JSONFileNameSeparator)] = JSONFileStats{
				CarrierID:         int32(carrierCode),
				FareFamiliesCount: len(fareFamiliesData),
			}
			storageInstance.data[int32(carrierCode)] = CarrierFareFamilies{
				CarrierID:    int32(carrierCode),
				FareFamilies: fareFamiliesCompiled,
			}
		} else {
			logger.Infof("Skipping: %s", path)
		}
		return nil
	})
	if err != nil {
		return nil, err
	}

	return storageInstance, nil
}

// For testing purposes only
func NewEmptyStorage(logger log.Logger) Storage {
	return &storageImpl{
		data:   map[int32]CarrierFareFamilies{},
		files:  map[string]JSONFileStats{},
		logger: logger,
	}
}

// For testing purposes only
func (s *storageImpl) AddFareFamily(carrier int32, fareFamily farefamiliesstructs.FareFamily) error {
	entry, ok := s.data[carrier]
	if !ok {
		entry = CarrierFareFamilies{
			CarrierID:    carrier,
			FareFamilies: []farefamiliesstructs.CompiledFareFamily{},
		}
	}
	compiledFareFamily, err := InitFareFamily(fareFamily)
	if err != nil {
		return err
	}
	entry.FareFamilies = append(entry.FareFamilies, *compiledFareFamily)
	s.data[carrier] = entry
	return nil
}

func (s *storageImpl) GetDump(carrier int32, filters ...FareFamilyFilterFunc) (dump *CarrierFareFamilies, err error) {
	carrierFares, ok := s.data[carrier]
	if !ok {
		s.logger.Info("Unknown carrier", log.Int32("carrier", carrier))
		return &CarrierFareFamilies{
			CarrierID:    carrier,
			FareFamilies: []farefamiliesstructs.CompiledFareFamily{},
		}, nil
	}
	result := CarrierFareFamilies{
		CarrierID:    carrierFares.CarrierID,
		FareFamilies: []farefamiliesstructs.CompiledFareFamily{},
	}
	for ffIndex, ff := range carrierFares.FareFamilies {
		if len(filters) == 0 {
			result.FareFamilies = append(result.FareFamilies, ff)
		} else {
			for _, filter := range filters {
				if filter(ff) {
					// TODO(u-jeen): calculate fare family key properly, via indices of each rule term
					ff.Key = fmt.Sprintf("ff_index=%d;carrier=%d", ffIndex, carrierFares.CarrierID)
					result.FareFamilies = append(result.FareFamilies, ff)
					break
				}
			}
		}
	}
	return &result, nil
}

func (s *storageImpl) GetFiles() map[string]JSONFileStats {
	result := map[string]JSONFileStats{}
	for k, v := range s.files {
		result[k] = v
	}
	return result
}

func InitFareFamily(ff farefamiliesstructs.FareFamily) (*farefamiliesstructs.CompiledFareFamily, error) {
	result := farefamiliesstructs.CompiledFareFamily{FareFamily: ff}
	if len(ff.NegatedTariffCodePattern) > 0 {
		compiledRegexp, err := regexp.Compile(ff.NegatedTariffCodePattern)
		if err != nil {
			return nil, err
		}
		result.TariffCodeRegExp = compiledRegexp
		result.NegateRegexp = true
	} else if len(ff.TariffCodePattern) > 0 {
		compiledRegexp, err := regexp.Compile(ff.TariffCodePattern)
		if err != nil {
			return nil, err
		}
		result.TariffCodeRegExp = compiledRegexp
	}

	for termIdx, term := range result.Terms {
		if term.Code == farefamiliesstructs.CarryOnCode {
			for ruleIdx, rule := range term.Rules {
				rule.CarryOnSizeBucket = GetCarryOnSizeBucket(rule.Size)
				term.Rules[ruleIdx] = rule
			}
			result.Terms[termIdx] = term
		}
	}
	return &result, nil
}
