package tariffmatcher

import (
	"strconv"

	farefamiliesstructs "a.yandex-team.ru/travel/avia/fare_families/internal/services/fare_families/data_structs/fare_families"
)

type chargeValue struct {
	availability farefamiliesstructs.Availability
	currency     string
	value        string
}

const (
	RUB = "RUB"
)

func unknownAmountCharge() chargeValue {
	return chargeValue{availability: farefamiliesstructs.AvailableForCharge}
}

func newChargeValue(term farefamiliesstructs.FareFamilyTermRule) chargeValue {
	result := chargeValue{availability: farefamiliesstructs.Availability(term.Availability)}
	if term.Charge != nil {
		result.currency = term.Charge.Currency
		result.value = term.Charge.Value
	}
	return result
}

func worseCharge(c1, c2 chargeValue) chargeValue {
	if c1.availability == farefamiliesstructs.NotAvailable {
		return c1
	}
	if c2.availability == farefamiliesstructs.NotAvailable {
		return c2
	}
	if len(c1.availability) == 0 {
		return c1
	}
	if len(c2.availability) == 0 {
		return c2
	}
	if c1.availability != farefamiliesstructs.AvailableForCharge {
		return c2
	}
	if c2.availability != farefamiliesstructs.AvailableForCharge {
		return c1
	}
	if len(c1.currency) == 0 {
		return c1
	}
	if len(c2.currency) == 0 {
		return c2
	}
	if c1.currency != RUB {
		return c1
	}
	if c2.currency != RUB {
		return c2
	}
	// at this point we have two RUB-based charges,
	// let's compare the amounts
	v1, err := strconv.Atoi(c1.value)
	if err != nil {
		return c2
	}

	v2, err := strconv.Atoi(c2.value)
	if err != nil {
		return c1
	}

	if v1 >= v2 {
		return c1
	}
	return c2
}

func addCharge(c1, c2 chargeValue) chargeValue {
	if c1.availability == farefamiliesstructs.NotAvailable {
		return c1
	}
	if c2.availability == farefamiliesstructs.NotAvailable {
		return c2
	}
	if len(c1.availability) == 0 {
		return c1
	}
	if len(c2.availability) == 0 {
		return c2
	}
	if c1.availability != farefamiliesstructs.AvailableForCharge {
		return c2
	}
	if c2.availability != farefamiliesstructs.AvailableForCharge {
		return c1
	}
	if len(c1.currency) == 0 {
		return c1
	}
	if len(c2.currency) == 0 {
		return c2
	}
	if c1.currency != c2.currency {
		return unknownAmountCharge()
	}
	// at this point we have two charges in the same currency,
	// let's add the amounts
	v1, err := strconv.Atoi(c1.value)
	if err != nil {
		return unknownAmountCharge()
	}

	v2, err := strconv.Atoi(c2.value)
	if err != nil {
		return unknownAmountCharge()
	}

	return chargeValue{
		availability: farefamiliesstructs.AvailableForCharge,
		currency:     c1.currency,
		value:        strconv.Itoa(v1 + v2),
	}
}

// Returns an updated rule and a suffix to append to the fare family key when replacing this rule
func (c *chargeValue) updateRule(rule farefamiliesstructs.FareFamilyTermRule) (farefamiliesstructs.FareFamilyTermRule, string) {
	ruleChargeIsSameAsChargeValue := rule.Availability == string(c.availability)
	if rule.Charge != nil && (rule.Charge.Currency != c.currency || rule.Charge.Value != c.value) {
		ruleChargeIsSameAsChargeValue = false
	}
	if ruleChargeIsSameAsChargeValue {
		return rule, ""
	}
	// rule charge is not the same --> let's erasr comments and everything except charge to make sure rule is not contradictory
	return farefamiliesstructs.FareFamilyTermRule{
			Availability: string(c.availability),
			Charge: &farefamiliesstructs.Charge{
				Currency: c.currency,
				Value:    c.value,
			},
		},
		c.getFareFamilyKeySuffix()
}

func (c *chargeValue) getFareFamilyKeySuffix() string {
	if c.availability == farefamiliesstructs.NotAvailable {
		return "not_avail"
	}
	if len(c.availability) == 0 {
		return "unknown"
	}
	if c.availability != farefamiliesstructs.AvailableForCharge {
		return string(c.availability)
	}
	return c.currency + c.value
}
