package tariffmatcher

import (
	"fmt"
	"regexp"
	"sync"

	lru "github.com/hashicorp/golang-lru"
)

const (
	MaxCacheSize = 1000000
)

type RegexpCache interface {
	MatchString(re *regexp.Regexp, tariffCode string) bool
}

type RegexpCacheImpl struct {
	cache *lru.Cache
	m     sync.RWMutex
}

type CacheValue struct {
	regexpString string
	tariffCode   string
}

func NewRegexpCache() RegexpCache {
	cache, err := lru.New(MaxCacheSize)
	if err != nil {
		panic(fmt.Sprintf("Unable to initialize regexp cache: %s", err.Error()))
	}
	return &RegexpCacheImpl{
		cache: cache,
		m:     sync.RWMutex{},
	}
}

func (rc *RegexpCacheImpl) MatchString(re *regexp.Regexp, tariffCode string) bool {
	valueToMatch := CacheValue{re.String(), tariffCode}
	result, ok := rc.getCacheValue(valueToMatch)
	if ok {
		return result
	}
	result = re.MatchString(tariffCode)
	go func() {
		rc.setCacheValue(valueToMatch, result)
	}()
	return result
}

func (rc *RegexpCacheImpl) getCacheValue(valueToMatch CacheValue) (result bool, ok bool) {
	rc.m.RLock()
	defer rc.m.RUnlock()
	cachedValue, okValue := rc.cache.Get(valueToMatch)
	if okValue {
		result = cachedValue.(bool)
	}
	return result, okValue
}

func (rc *RegexpCacheImpl) setCacheValue(valueToMatch CacheValue, result bool) {
	rc.m.Lock()
	defer rc.m.Unlock()
	rc.cache.Add(valueToMatch, result)
}

// For testing purposes only
func (rc *RegexpCacheImpl) getCachedKeys() []CacheValue {
	rc.m.Lock()
	defer rc.m.Unlock()
	result := []CacheValue{}
	for _, key := range rc.cache.Keys() {
		result = append(result, key.(CacheValue))
	}
	return result
}
