package main

import (
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/indexworker"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/library/go/logging"
	"context"
	"encoding/json"
	"fmt"
	"github.com/go-chi/chi/v5"
	"net/http"

	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/configprovider"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/controllers"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/dataprovider"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/db"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/environmentvariableprovider"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/indexbuilder"
	"a.yandex-team.ru/travel/library/go/syncutil"

	arclog "a.yandex-team.ru/library/go/core/log"
)

func main() {
	ctx, cancelFunc := context.WithCancel(context.Background())

	config, err := configprovider.New(
		environmentvariableprovider.New(),
	).GetConfig()
	if err != nil {
		panic(err)
	}

	logger, err := logging.NewDeploy(config.Logging)
	if err != nil {
		panic(fmt.Errorf("couldn't create application logger. Error: %+v", err))
	}

	var wg syncutil.WaitGroup

	instanceIndexBuilder := indexbuilder.New(
		db.New(),
		dataprovider.NewServiceProvider(),
		dataprovider.NewFlagProvider(),
		dataprovider.NewRelationsProvider(),
		config,
	)

	indexWorker := indexworker.NewIndexWorker(logger, config.IndexWorker, instanceIndexBuilder)
	wg.Go(func() { indexWorker.Work(ctx) })

	router := chi.NewRouter()
	router.Get("/feature-flag", func(w http.ResponseWriter, r *http.Request) {
		params := r.URL.Query()
		status, data := controllers.GetByCode(indexWorker, params)
		w.WriteHeader(status)
		_ = json.NewEncoder(w).Encode(data)
	})

	srv := &http.Server{
		Addr:    ":" + config.ListenPort,
		Handler: router,
	}

	go func() {
		err := srv.ListenAndServe()
		if err != nil && err != http.ErrServerClosed {
			logger.Fatal("", arclog.Error(err))
		}
	}()

	probeService := probes.NewState(
		logger,
		probes.OnReady(indexWorker.Ping),
		probes.OnStopBefore(cancelFunc),
		probes.OnStopAfter(indexWorker.WaitStop),
	)
	probes.ChiBind(config.Probes, router, probeService)

	wg.Wait()

	err = srv.Shutdown(context.Background())
	if err != nil {
		logger.Error("", arclog.Error(err))
	}
}
