package configprovider

import (
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/library/go/logging"
	"fmt"
	"strconv"
	"time"

	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/baseerror"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/environmentvariableprovider"
)

type MySQLConfig struct {
	ReadHosts []string
	UserName  string
	Password  string
	Protocol  string
	Port      int
	Schema    string
}

type IndexWorkerConfig struct {
	SuccessSleep time.Duration
	FailSleep    time.Duration
}

type Config struct {
	Environment string
	ListenPort  string
	Mysql       *MySQLConfig
	Probes      *probes.HTTPConfig
	IndexWorker *IndexWorkerConfig
	Logging     *logging.Config
}

type ConfigProvider struct {
	environmentProvider environmentvariableprovider.IEnvironmentVariableProvider
}

type IConfigProvider interface {
	GetConfig() (*Config, error)
}

func New(environmentVariableProvider environmentvariableprovider.IEnvironmentVariableProvider) IConfigProvider {
	return &ConfigProvider{
		environmentVariableProvider,
	}
}

func (provider *ConfigProvider) GetConfig() (*Config, error) {
	environment, err := provider.environmentProvider.ReadRequiredEnv("ENVIRONMENT")
	if err != nil {
		return nil, err
	}
	dbHost, err := provider.environmentProvider.ReadRequiredEnv("DB_HOST")
	if err != nil {
		return nil, err
	}

	rawPort, err := provider.environmentProvider.ReadRequiredEnv("DB_PORT")
	if err != nil {
		return nil, err
	}

	port, err := strconv.Atoi(rawPort)
	if err != nil {
		return nil, &baseerror.BaseError{
			Message: fmt.Sprintf(
				"Can not parse port %v",
				rawPort,
			)}
	}

	dbUserName, err := provider.environmentProvider.ReadRequiredEnv("DB_USER_NAME")
	if err != nil {
		return nil, err
	}
	dbProtocol, err := provider.environmentProvider.ReadRequiredEnv("DB_PROTOCOL")
	if err != nil {
		return nil, err
	}
	dbPassword, err := provider.environmentProvider.ReadRequiredEnv("DB_PASSWORD")
	if err != nil {
		return nil, err
	}
	dbSchema, err := provider.environmentProvider.ReadRequiredEnv("DB_SCHEMA")
	if err != nil {
		return nil, err
	}

	listenPort := provider.environmentProvider.ReadEnv("LISTEN_PORT")
	if listenPort == "" {
		listenPort = "80"
	}

	successSleepSeconds := provider.environmentProvider.ReadEnv("INDEX_WORKER_SUCCESS_SLEEP_SECONDS")
	if successSleepSeconds == "" {
		successSleepSeconds = "60"
	}
	successSleep, err := strconv.Atoi(successSleepSeconds)
	if err != nil {
		return nil, err
	}

	failSleepSeconds := provider.environmentProvider.ReadEnv("INDEX_WORKER_FAIL_SLEEP_SECONDS")
	if failSleepSeconds == "" {
		failSleepSeconds = "300"
	}
	failSleep, err := strconv.Atoi(failSleepSeconds)
	if err != nil {
		return nil, err
	}

	return &Config{
		Environment: environment,
		ListenPort:  listenPort,
		Mysql: &MySQLConfig{
			ReadHosts: []string{dbHost},
			UserName:  dbUserName,
			Protocol:  dbProtocol,
			Password:  dbPassword,
			Port:      port,
			Schema:    dbSchema,
		},
		Probes: &probes.DefaultHTTPConfig,
		IndexWorker: &IndexWorkerConfig{
			SuccessSleep: time.Second * time.Duration(successSleep),
			FailSleep:    time.Second * time.Duration(failSleep),
		},
		Logging: &logging.DefaultConfig,
	}, err
}
