package indexbuilder

import (
	"fmt"
	"time"

	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/configprovider"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/dataprovider"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/db"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/models"
)

type Index struct {
	ServiceByCode      map[string]*models.Service
	ServiceByID        map[int]*models.Service
	FlagByID           map[int]*models.FeatureFlag
	FlagByCode         map[string]*models.FeatureFlag
	FlagIdsByServiceID map[int]map[int]bool
	LastUpdate         time.Time
}

type Builder struct {
	connector         db.IConnector
	serviceProvider   dataprovider.IServiceProvider
	flagProvider      dataprovider.IFlagProvider
	relationsProvider dataprovider.IRelationsProvider
	config            *configprovider.Config
}

type IBuilder interface {
	FetchAndBuild(config *configprovider.Config) (*Index, error)
	Build(services []*models.Service, flags []*models.FeatureFlag, relations []*models.ServiceFeatureFlagRelation) *Index
}

func New(connector db.IConnector,
	serviceProvider dataprovider.IServiceProvider,
	flagProvider dataprovider.IFlagProvider,
	relationsProvider dataprovider.IRelationsProvider,
	config *configprovider.Config) *Builder {
	return &Builder{
		connector,
		serviceProvider,
		flagProvider,
		relationsProvider,
		config,
	}
}

func (b *Builder) FetchAndBuild() (*Index, error) {
	con, err := b.connector.Connect(&db.ConnectionConfig{
		Slaves:   b.config.Mysql.ReadHosts,
		UserName: b.config.Mysql.UserName,
		Password: b.config.Mysql.Password,
		Protocol: b.config.Mysql.Protocol,
		Port:     b.config.Mysql.Port,
		Schema:   b.config.Mysql.Schema,
	})
	if err != nil {
		return nil, fmt.Errorf("connecting: %+v", err)
	}
	if con != nil {
		defer con.Close()
	}

	services, err := b.serviceProvider.Fetch(con)
	if err != nil {
		return nil, fmt.Errorf("service fetching: %+v", err)
	}

	flags, err := b.flagProvider.Fetch(con)
	if err != nil {
		return nil, fmt.Errorf("flag fetching: %+v", err)
	}

	relations, err := b.relationsProvider.Fetch(con)
	if err != nil {
		return nil, fmt.Errorf("relation fetching: %+v", err)
	}

	return b.Build(services, flags, relations), nil
}

func (b *Builder) Build(services []*models.Service, flags []*models.FeatureFlag, relations []*models.ServiceFeatureFlagRelation) *Index {
	serviceByCode := make(map[string]*models.Service)
	serviceByID := make(map[int]*models.Service)

	for _, service := range services {
		serviceByID[service.ID] = service
		serviceByCode[service.Code] = service
	}

	flagByID := make(map[int]*models.FeatureFlag)
	flagByCode := make(map[string]*models.FeatureFlag)

	for _, flag := range flags {
		flagByID[flag.ID] = flag
		flagByCode[flag.Code] = flag
	}

	flagIdsByServiceID := make(map[int]map[int]bool)

	for _, r := range relations {
		_, hasService := serviceByID[r.ServiceID]
		_, hasFeatureFlag := flagByID[r.FeatureFlagID]
		if hasService && hasFeatureFlag {
			_, hasFlags := flagIdsByServiceID[r.ServiceID]
			if !hasFlags {
				flagIdsByServiceID[r.ServiceID] = make(map[int]bool)
			}
			flagIdsByServiceID[r.ServiceID][r.FeatureFlagID] = true
		}
	}

	return &Index{
		serviceByCode,
		serviceByID,
		flagByID,
		flagByCode,
		flagIdsByServiceID,
		time.Now(),
	}
}
