package indexworker

import (
	"context"
	"fmt"
	"sync"
	"time"

	arclog "a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/configprovider"
	"a.yandex-team.ru/travel/avia/feature_flag_api/internal/indexbuilder"
)

type IndexWorker struct {
	cfg                  *configprovider.IndexWorkerConfig
	instanceIndexBuilder *indexbuilder.Builder
	logger               *arczap.Logger

	confirmStop  chan bool
	currentIndex *indexbuilder.Index
	rwMutex      sync.RWMutex
}

func NewIndexWorker(logger *arczap.Logger, cfg *configprovider.IndexWorkerConfig, instanceIndexBuilder *indexbuilder.Builder) *IndexWorker {
	return &IndexWorker{
		cfg:                  cfg,
		instanceIndexBuilder: instanceIndexBuilder,
		logger:               logger,

		confirmStop:  make(chan bool, 1),
		currentIndex: nil,
	}
}

func (iw *IndexWorker) Work(ctx context.Context) {
	timeToSleep := time.Duration(0)

	running := true

	for running {
		timer := time.NewTimer(timeToSleep)
		iw.logger.Infof("Sleep %v", timeToSleep)

		select {
		case <-ctx.Done():
			timer.Stop()
			running = false
		case <-timer.C:
			newIndex, err := iw.instanceIndexBuilder.FetchAndBuild()

			if err != nil {
				iw.logger.Error("Fetch error", arclog.Error(err))
				timeToSleep = iw.cfg.FailSleep
			} else {
				iw.setIndex(newIndex)
				timeToSleep = iw.cfg.SuccessSleep
			}
		}
	}

	iw.confirmStop <- true
}

func (iw *IndexWorker) Ping() error {
	if iw.GetCurrentIndex() == nil {
		return fmt.Errorf("index not ready")
	}
	return nil
}

func (iw *IndexWorker) WaitStop() {
	<-iw.confirmStop
}

func (iw *IndexWorker) setIndex(index *indexbuilder.Index) {
	iw.rwMutex.Lock()
	iw.currentIndex = index
	iw.rwMutex.Unlock()
}

func (iw *IndexWorker) GetCurrentIndex() *indexbuilder.Index {
	iw.rwMutex.RLock()
	index := iw.currentIndex
	iw.rwMutex.RUnlock()
	return index
}
