import logging

import requests
from raven.contrib.tornado import AsyncSentryClient
from tornado.web import Application

from travel.avia.flight_extras.application.cache import cache
from travel.avia.flight_extras.application.handlers import PingHandler, VersionHandler
from travel.avia.flight_extras.application.handlers.flight import FlightHandler
from travel.avia.flight_extras.application.handlers.flight_passenger_experience import UpdateAircraftHandler
from travel.avia.flight_extras.application.handlers.flights_list import FlightsListHandler
from travel.avia.flight_extras.settings import SENTRY_URL, AVIA_BACKEND_URL, DEBUG


RASP_SOURCE_NAME = '_travel.rasp_'


logger = logging.getLogger()
if DEBUG:
    logging.getLogger('sqlalchemy.engine').setLevel(logging.INFO)


class FlightExtrasApplication(Application):
    def pre_cache(self):
        logging.info('Precaching...')
        self.pre_cache_planes()
        logging.info('Precached')

    @staticmethod
    def pre_cache_planes():
        logging.info('Precaching planes...')
        cache.set('planes', {})

        response = requests.get(
            '{}/rest/transport_model/planes'.format(AVIA_BACKEND_URL),
            verify=False,
            timeout=5,
        )
        if response.status_code != 200:
            raise RuntimeError('Error precaching planes: %s' % response.reason)

        result = response.json()
        if result['status'] != 'ok':
            raise RuntimeError('Error precaching planes: %s' % result['data'])

        cache.set('planes', {
            p['code_en']: p for p in result['data'] if p['code_en']
        })


def create_application(debug=False):
    application = FlightExtrasApplication(
        [
            (r'/ping', PingHandler),
            (r'/version', VersionHandler),
            (r'/flight/([^/]+)/([^/]+)/(\d{4}\-\d{2}\-\d{2})/', FlightHandler),
            (r'/flights/', FlightsListHandler),

            (r'/flight/([^/]+)/([^/]+)/aircraft/', UpdateAircraftHandler),
        ],
        debug=debug
    )

    application.sentry_client = AsyncSentryClient(SENTRY_URL)
    application.pre_cache()

    return application
