from raven.contrib.tornado import SentryMixin
from sqlalchemy import desc, func
from sqlalchemy.orm import contains_eager
from tornado.web import RequestHandler

from travel.avia.flight_extras.application import db
from travel.avia.flight_extras.application.cache import cache
from travel.avia.flight_extras.application.models import Flight, FlightPassengerExperience, FlightInfo


class FlightHandler(SentryMixin, RequestHandler):
    def get(self, company_iata, number, departure_day):
        # type: (str, str, str) -> None
        session = db.db_slave.create_session()

        query = session.query(
            FlightPassengerExperience,
        ).join(
            FlightPassengerExperience.flight,
            FlightPassengerExperience.source,
        ).options(
            contains_eager(FlightPassengerExperience.flight),
            contains_eager(FlightPassengerExperience.source),
        ).filter(
            Flight.company_iata == company_iata.upper(),
            Flight.number == number,
        ).order_by(
            desc(FlightPassengerExperience.departure_day)
        )

        if self.get_argument('use_approximation', 0) == '1':
            query = query.filter(
                FlightPassengerExperience.departure_day <= departure_day,
                func.mod(FlightPassengerExperience.departure_day - departure_day, 7) == 0
            )
        else:
            query = query.filter(
                FlightPassengerExperience.departure_day == departure_day,
            )

        flight = query.first() or FlightPassengerExperience()

        result = flight.as_dict()
        if result['aircraft']:
            result['aircraft'] = cache.get('planes').get(result['aircraft'])

        if flight.id and 'extra_info' in self.get_argument('fields', []):
            extra_info = session.query(FlightInfo).filter(
                FlightInfo.flight_id == flight.flight_id,
                FlightInfo.departure_day == departure_day,
            ).first()

            result['extra_info'] = extra_info.as_dict() if extra_info else None

        self.write(result)
