from contextlib import closing
from datetime import datetime

import ujson
from raven.contrib.tornado import SentryMixin
from tornado.web import RequestHandler

from travel.avia.flight_extras.application.db import db_master
from travel.avia.flight_extras.application.library.flight_aircraft_updater import FlightAircraftUpdater
from travel.avia.flight_extras.application.models import Flight, FlightPassengerExperience, Source


class UpdateAircraftHandler(SentryMixin, RequestHandler):
    def post(self, company_iata, number):
        # type: (str, str) -> None

        try:
            data = ujson.loads(self.request.body)
        except (ValueError, KeyError):
            self.set_status(400)
            self.finish('Malformed json')
            return

        source = data.get('source')
        if not source:
            self.set_status(400)
            self.finish('Empty source')
            return

        with closing(db_master.create_session()) as session:
            updater = FlightAircraftUpdater(session)

            s = Source.get_or_create(source, session)
            flight = Flight(
                company_iata=company_iata,
                number=number,
            )

            for d in data.get('data', []):
                aircraft = d.get('aircraft')
                departure_day = d.get('departure_day')
                if not aircraft or not departure_day:
                    continue
                pe = FlightPassengerExperience(
                    flight=flight,
                    source_id=s.id,
                    departure_day=datetime.strptime(d['departure_day'], '%Y-%m-%d').date(),
                    aircraft=d['aircraft'],
                )
                updater.add_flight_passenger_experience(pe)

            updater.flush()
            session.commit()

        self.write('Updated: {}'.format(updater.count))
