from collections import namedtuple
from datetime import datetime

from raven.contrib.tornado import SentryMixin
from sqlalchemy import or_
from sqlalchemy.orm import contains_eager
from tornado.web import RequestHandler, HTTPError

from travel.avia.flight_extras.application import db
from travel.avia.flight_extras.application.cache import cache
from travel.avia.flight_extras.application.models import Flight, FlightPassengerExperience

DEPARTURE_DATE_FORMAT = '%Y-%m-%d'
FlightKey = namedtuple('FlightKey', ['departure_date', 'iata', 'number'])


def parse_flight_key(flight):
    # type: (str) -> FlightKey
    return FlightKey(
        departure_date=datetime.strptime(flight[:8], '%Y%m%d').strftime(DEPARTURE_DATE_FORMAT),
        iata=flight[8:10],
        number=flight[10:],
    )


def get_filtered_flights(db_session, flight_requests):
    return (
        db_session.query(
            FlightPassengerExperience,
        ).join(
            FlightPassengerExperience.flight,
            FlightPassengerExperience.source,
        ).options(
            contains_eager(FlightPassengerExperience.flight),
            contains_eager(FlightPassengerExperience.source),
        ).filter(
            or_(
                *[
                    (Flight.number == f.number) &
                    (Flight.company_iata == f.iata) &
                    (FlightPassengerExperience.departure_day == f.departure_date)
                    for f in flight_requests
                ]
            )
        ).order_by(
            Flight.id.desc(),
            FlightPassengerExperience.departure_day.desc(),
        ).distinct(
            Flight.id, FlightPassengerExperience.departure_day,
        )
    )


def get_latest_version_flights(flights, flight_requests):
    flight_requests_set = set(flight_requests)

    latest_version_flights = {}
    for f in flights:
        flight_key = FlightKey(
            f.departure_day.strftime(DEPARTURE_DATE_FORMAT),
            f.flight.company_iata,
            f.flight.number
        )
        if flight_key in flight_requests_set:
            flight_dict = f.as_dict()
            if flight_dict['aircraft']:
                flight_dict['aircraft'] = cache.get('planes').get(flight_dict['aircraft'])
            latest_version_flights[flight_key] = flight_dict
    return latest_version_flights


def parse_flight_requests(flights_argument):
    for f in flights_argument.split(','):
        try:
            yield parse_flight_key(f)
        except ValueError:
            raise HTTPError(
                400,
                'Flight "{}" has incorrect format. Expected format: %Y%m%d{{iata}}{{number}}'.format(f)
            )


class FlightsListHandler(SentryMixin, RequestHandler):
    def get(self):
        self.handle_request(self.get_argument('flights'))

    def post(self):
        self.handle_request(self.get_body_argument('flights'))

    def handle_request(self, flights):
        if not flights:
            raise HTTPError(400, 'Missing argument flights')

        flight_requests = list(parse_flight_requests(flights))
        session = db.db_slave.create_session()
        filtered_flights = get_filtered_flights(session, flight_requests)
        latest_version_flights = get_latest_version_flights(filtered_flights, flight_requests)

        self.write({
            'data': [
                latest_version_flights.get(f)
                for f in flight_requests
            ],
        })
