import logging
import pytz
import requests
from datetime import datetime
from typing import Optional

from travel.avia.flight_extras.settings import SKY_GURU_API_URL, SKY_GURU_AUTH_TOKEN

logger = logging.getLogger(__name__)


class SkyGuru(object):
    TIMEOUT = 180

    def __init__(self, api_url, auth_token):
        # type: (str, str) -> None
        if not api_url:
            raise RuntimeError('Empty SkyGuru api url')
        self._api_url = api_url
        self._auth_token = auth_token

    def request(self, path, params=None):
        # type: (str, Optional[dict]) -> Optional[dict]
        try:
            headers = {
                'App-Token': self._auth_token,
            }
            request_url = '{api_url}{method}'.format(
                api_url=self._api_url,
                method=path,
            )

            logger.info('Request: "%s" with %s', request_url, params)

            response = requests.get(
                url=request_url,
                headers=headers,
                params=params,
                timeout=self.TIMEOUT,
            )

            if response.status_code == 404:
                return None

            response.raise_for_status()

            return response.json()

        except Exception as e:
            logger.exception('Error request sky guru: %s', e)

        return None

    def get_flight(self, company_iata, number, departure):
        # type: (str, str, datetime) -> dict
        return self.request('/flights', {
            'flight_date': departure.astimezone(pytz.UTC).strftime('%Y-%m-%dT%H:%M:%SZ'),
            'flight_number': company_iata + number,
            'utc_offset': departure.utcoffset().total_seconds() / 3600,
        })

    def get_route(self, flight_id):
        # type: (str) -> dict
        return self.request('/flights/{}/zones'.format(flight_id))

    def get_sun(self, flight_id):
        # type: (str) -> dict
        return self.request('/flights/{}/sun'.format(flight_id))

    def get_wind(self, flight_id):
        # type: (str) -> dict
        return self.request('/flights/{}/wind'.format(flight_id))

    def get_weather(self, flight_id):
        # type: (str) -> dict
        return self.request('/flights/{}/weather'.format(flight_id))

    def get_sights(self, flight_id):
        # type: (str) -> dict
        return self.request('/flights/{}/sights'.format(flight_id))

    def get_aircraft(self, aircraft_id):
        # type: (str) -> dict
        return self.request('/aircrafts/{}'.format(aircraft_id))

    def get_aircraft_extra_info(self, aircraft_id):
        # type: (str) -> dict
        return self.request('/aircrafts/{}/info'.format(aircraft_id))

    def get_airport(self, airport_id):
        # type: (str) -> dict
        return self.request('/airports/{}'.format(airport_id))


sky_guru = SkyGuru(
    SKY_GURU_API_URL,
    SKY_GURU_AUTH_TOKEN,
)
