# coding=utf-8
from __future__ import absolute_import

import errno
import logging
from ciso8601 import parse_datetime
from datetime import datetime, timedelta

import os
import os.path
import pytz
import re
import requests
from furl import furl

from travel.avia.flight_extras.application.models import Source
from travel.avia.flight_extras.bin.upload_flight_passenger_experience import upload_from_file, get_previous_filename_and_datetime
from travel.avia.flight_extras.settings import setup_logging

PASSENGER_EXPERIENCE_FILES_URL = 'https://api.flightstats.com/flex/datadrop/rest/v1/json/list/'
APP_ID = os.getenv('PASSENGER_EXPERIENCE_APP_ID')
APP_KEY = os.getenv('PASSENGER_EXPERIENCE_APP_KEY')
FILES_FOLDER = os.getenv('PASSENGER_EXPERIENCE_DIRECTORY')
CHUNK_SIZE = 2 ** 21  # 2 MB

RE_DATA_URL = re.compile(r'^(http.+' + Source.FLIGHT_STATS_PREFIX + r'\d+\.csv)$')

log = logging.getLogger('fetch_and_upload_flight_passenger_experience')


def with_auth_info(url):
    new_url = furl(url)
    new_url.query.params.update({'appId': APP_ID, 'appKey': APP_KEY})
    return new_url.url


def download_to_file(url, filename):
    response = requests.get(url, stream=True)
    with open(filename, 'w') as f:
        for data in response.iter_content(chunk_size=CHUNK_SIZE):
            f.write(data)


def get_latest_url():
    files_info = requests.get(with_auth_info(PASSENGER_EXPERIENCE_FILES_URL)).json()
    files = filter(lambda info: RE_DATA_URL.match(info['uri']), files_info['files'])
    return max(files, key=lambda f: parse_datetime(f['modified']))['uri']


def make_dirs(path):
    try:
        os.makedirs(path)
    except OSError as e:
        if e.errno != errno.EEXIST:
            raise


def main():
    setup_logging()

    if os.getenv('YANDEX_ENVIRONMENT_TYPE') not in ('development', 'testing', 'production'):
        log.info('Allow only in %s', ('development', 'testing', 'production'))
        return

    data_url = get_latest_url()
    log.info('Latest url: %s', with_auth_info(data_url))
    make_dirs(FILES_FOLDER)
    passenger_experience_filename = os.path.join(FILES_FOLDER, re.search(r'/([^/]+\.csv)', data_url).group(1))

    log.info('Filename for passenger_experience: %s', passenger_experience_filename)
    prev_filename, prev_datetime = get_previous_filename_and_datetime()
    if prev_filename == os.path.basename(passenger_experience_filename):
        now = pytz.utc.localize(datetime.utcnow())
        if now - prev_datetime > timedelta(days=40):
            raise Exception("File wasn't updated for 40 days in a row.")
        log.info('Latest file already loaded to database.')
        return
    try:
        log.info('Got new file. Start downloading.')
        download_to_file(with_auth_info(data_url), passenger_experience_filename)
        log.info('Loaded file. Start updating database.')
        upload_from_file(passenger_experience_filename)
        log.info('Database successfully updated.')
    finally:
        if os.path.exists(passenger_experience_filename):
            os.remove(passenger_experience_filename)


if __name__ == '__main__':
    main()
